# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import logging
import six

from django import forms
from django.core import validators as django_validators
from django.core.exceptions import ValidationError
from django.utils.text import Truncator
from django.utils.translation import gettext_lazy as _

from app.utils.misc import blank
from core.forms import YandexFormMixin, NewPaymentInfoForm as BaseNewPaymentInfoForm
from core.misc import Captcha
from core.models import PaymentInfo
from core.models.user import NewPaymentInfo
from core import validators


logger = logging.getLogger(__name__)


class Textarea(forms.Textarea):
    """Set default attrs to None."""

    def __init__(self, attrs=None):
        super(Textarea, self).__init__(attrs)


class CharField(forms.CharField):
    """CharField with custom widget attrs."""

    def widget_attrs(self, widget):
        attrs = super(CharField, self).widget_attrs(widget)
        attrs.update({
            'autocomplete': 'off',
            'size': '1',
            'class': 'input__input',
        })
        return attrs


class TextField(forms.CharField):
    """TextField with custom widget attrs."""
    widget = Textarea

    def widget_attrs(self, widget):
        return {
            'rows': '9',
            'class': 'textarea__textarea',
        }


class URLField(CharField):
    default_validators = [django_validators.URLValidator()]


class DocumentNumberWidget(forms.TextInput):

    def _format_value(self, value):
        return self.mask_value(value)

    def mask_value(self, value):
        if value is None or len(value) <= 4:
            return value
        head = value[:2]
        body = value[2:-2]
        tail = value[-2:]
        truncated = Truncator(body).chars(2, truncate='%(truncated_text)s**')
        return ''.join([head, truncated, tail])


class PaymentInfoForm(YandexFormMixin, forms.ModelForm):
    captcha_key = forms.CharField()
    captcha = forms.CharField()
    document = forms.FileField(required=False)
    iban = forms.CharField(max_length=34, required=False,
                           validators=[validators.IBANValidator()])
    swift_code = forms.CharField(max_length=13, required=False,
                                 validators=[validators.SWIFTValidator()])
    ru_bic_code = forms.CharField(max_length=9, required=False,
                                  validators=[validators.BICValidator()])
    date_of_birth = forms.DateField()
    account_address = forms.CharField(widget=forms.Textarea)
    document_number = forms.CharField(max_length=31)
    document_issued = forms.CharField(widget=forms.Textarea)
    ru_document_issuer_code = forms.CharField(required=False, max_length=31)

    class Meta:
        model = PaymentInfo
        exclude = [
            'reporter',
            'mdm_ticket',
            'is_actual',
        ]
        widgets = {
            'document_number': DocumentNumberWidget,
        }

    def __init__(self, *args, **kwargs):
        super(PaymentInfoForm, self).__init__(*args, **kwargs)
        self._init_placeholders()

    def _init_placeholders(self):
        placeholders = {
            'iban': _('iban.iban_format'),
            'swift_code': _('swift.swift_format'),
            'ru_bic_code': _('payment_info.bic_format'),
            'date_of_birth': _('payment_info.date_format'),
            'ru_document_issuer_code': _('payment_info.code_format'),
        }
        for name, placeholder in placeholders.items():
            self.fields[name].widget.attrs['placeholder'] = placeholder

    def clean(self):
        cleaned_data = super(PaymentInfoForm, self).clean()
        self._clean_captcha(cleaned_data)
        if cleaned_data['is_russian_resident']:
            self._clean_russian(cleaned_data)
            self._russian_required_fields()
        else:
            self._clean_foreign(cleaned_data)
            self._foreign_required_fields()
        return self.cleaned_data

    def _clean_captcha(self, cleaned_data):
        captcha_value = cleaned_data.get('captcha', None)
        captcha_key = cleaned_data.get('captcha_key', None)
        captcha = Captcha(captcha_key)
        if captcha.is_invalid(captcha_value):
            self.add_error('captcha', _('captcha.is_not_valid'))

    def _clean_russian(self, cleaned_data):
        pass

    def _russian_required_fields(self):
        for field_name in ('ru_fio',
                           'ru_account_number',
                           'ru_bank_name',
                           'ru_correspond_number',
                           'ru_bic_code'):
            self._check_required_fields(field_name)

    def _clean_foreign(self, cleaned_data):
        iban = cleaned_data.get('iban')
        swift_code = cleaned_data.get('swift_code')
        if not (iban or swift_code):
            self.add_error('swift_code', _('iban or swift are required'))
            self.add_error('iban', _('iban or swift are required'))
            raise forms.ValidationError(_('iban or swift are required'))

    def _foreign_required_fields(self):
        for field_name in ('account_holder',
                           'account_number',
                           'bank_name'):
            self._check_required_fields(field_name)

    def _check_required_fields(self, field_name):
        field = self.fields.get(field_name)
        value = self.cleaned_data.get(field_name)
        if value in field.empty_values:
            message = field.error_messages['required']
            error = ValidationError(message, code='required')
            self.add_error(field_name, error)


class NewPaymentInfoForm(BaseNewPaymentInfoForm):
    captcha_key = forms.CharField()
    captcha = forms.CharField()

    def _init_placeholders(self):
        super(NewPaymentInfoForm, self)._init_placeholders()
        if isinstance(self.instance, NewPaymentInfo):
            for datasync_key in self.instance.DATASYNC_FIELDS:
                if datasync_key in ('type', 'id', 'document'):
                    continue
                field_value = self.instance.datasync_values.get(datasync_key)
                if field_value:
                    field_name = datasync_key.replace('-', '_')
                    self.fields[field_name].widget.attrs['placeholder'] = blank(field_value)

    def _set_already_filled_fields(self):
        # Тут формат джанговых форм – через _
        # Получаем поля из datasync, возвращаем множество заполненных полей
        # и множество полей, заполненных латиницей
        self.filled_fields = set()
        self.filled_latin_fields = set()

        if not self.instance:
            return
        for datasync_key, value in self.instance.datasync_values.items():
            if datasync_key in ('type', 'id'):
                continue
            field_name = datasync_key.replace('-', '_')
            field = self.base_fields.get(field_name)
            if field and value not in field.empty_values:
                self.filled_fields.add(field_name)
                if not isinstance(value, six.text_type) or all([ord(ch) < 128 for ch in value]):
                    self.filled_latin_fields.add(field_name)

    def _clean_captcha(self, cleaned_data):
        captcha_value = cleaned_data.get('captcha', None)
        captcha_key = cleaned_data.get('captcha_key', None)
        captcha = Captcha(captcha_key)
        if captcha.is_invalid(captcha_value):
            self.add_error('captcha', _('captcha.is_not_valid'))
