# -*- coding: utf-8 -*-
from __future__ import absolute_import
from __future__ import unicode_literals

import logging

import magic

from django.conf import settings
from django.core import mail
from django.template import loader
from django.template import Context

from core.misc import generate_attachment_name


logger = logging.getLogger(__name__)


class OTRSMessage(object):
    template_name = None

    def __init__(self, yauser, message_context, language=None):
        super(OTRSMessage, self).__init__()
        self._language = language
        self._username = yauser.login
        self._message = self._init_message(message_context)
        self._message.from_email = yauser.default_email
        self._message.to = self.get_to_address_list()
        self._message.extra_headers = {
            'X-OTRS-fromfeedback': 'security',
            settings.OTRS_UID_HEADER: yauser.uid,
        }

    def _init_message(self, message_context):
        template = loader.get_template(self.get_template_name())
        message = template.render(Context(message_context))
        return mail.EmailMessage(body=message)

    def get_to_address_list(self):
        return [self.to_address]

    def get_template_name(self):
        return self.template_name

    def send(self):
        self._message.send()
        logger.info('Message from %s to %s was sent',
                    self._message.from_email,
                    ', '.join(self._message.to))

    def attach(self, attachment):
        content = attachment.read()
        mimetype = magic.from_buffer(content, mime=True)
        filename = generate_attachment_name(self._username, mimetype)
        self._message.attach(filename, content, mimetype)


class OTRSReportMessage(OTRSMessage):
    template_name = 'email/report_message.txt'

    def get_to_address_list(self):
        return [settings.OTRS_EMAILS[self._language]]

    def _init_message(self, message_context):
        message = super(OTRSReportMessage, self)._init_message(message_context)
        message.subject = message_context.pop('subject')
        return message


class OTRSPaymentInfoMessage(OTRSMessage):
    subject = 'Request for changing payment info'
    to_address = settings.OTRS_EMAIL_PAYMENT

    def __init__(self, yauser, message_context, language=None):
        self._is_russian_resident = message_context.get('is_russian_resident')
        super(OTRSPaymentInfoMessage, self).__init__(
            yauser, message_context, language)

    def _init_message(self, message_context):
        message = super(OTRSPaymentInfoMessage, self)._init_message(message_context)
        message.subject = self.subject
        return message

    def get_template_name(self):
        prefix = 'ru' if self._is_russian_resident else 'en'
        return '%s_edit_payment_request.html' % prefix
