# -*- coding: utf-8 -*-

import urllib

from django.conf import settings
from django.views import generic as views
from django.utils.decorators import method_decorator

from app import decorators as view_decorators
from app.views import mixins
from core import models
from core.utils import validator


class EditablePageView(mixins.MenuMixin,
                       mixins.EditablePageMixin,
                       views.DetailView):
    queryset = models.PageTranslation.published
    slug_field = 'page__slug'
    template_name = ''


class MainRedirect(EditablePageView):
    template_name = 'redirect.html'

    def get_context_data(self, **kwargs):
        render_context = 'open_redirect'
        if self.request.LANGUAGE_CODE=="ru":
            url = settings.PLC_REDIRECT_DOMAIN.format('ru')
        else:
            url = settings.PLC_REDIRECT_DOMAIN.format('com')
        return super(MainRedirect, self).get_context_data(
            render_context=render_context, url=url,
            **kwargs
        )


class ReportView(EditablePageView):
    template_name = 'report.html'

    def get_context_data(self, **kwargs):
        return super(ReportView, self).get_context_data(
            report_form_url=self._get_form_url(),
            report_form_name=self._get_form_name(),
            **kwargs
        )

    def _get_form_url(self):
        if self.request.LANGUAGE_CODE == 'ru':
            base_form_url = settings.REPORT_FORM_URL_RU
        else:
            base_form_url = settings.REPORT_FORM_URL_EN
        form_url = base_form_url + urllib.urlencode({
            'iframe': 1,
            'lang': self.request.LANGUAGE_CODE,
        })
        return form_url

    def _get_form_name(self):
        if self.request.LANGUAGE_CODE == 'ru':
            form_name = settings.REPORT_FORM_NAME_RU
        else:
            form_name = settings.REPORT_FORM_NAME_EN
        return form_name

    @method_decorator(view_decorators.yalogin_required_redirect(redirect_url=settings.LOGIN_URL))
    @method_decorator(view_decorators.email_required_redirect(redirect_url='email-required'))
    def dispatch(self, request, *args, **kwargs):
        return super(ReportView, self).dispatch(request, *args, **kwargs)


class LoginRequired(EditablePageView):
    template_name = 'redirect.html'
    

    def get_context_data(self, **kwargs):
        render_context = 'login'
        return super(LoginRequired, self).get_context_data(
            render_context=render_context,
            **kwargs
        )

    @method_decorator(view_decorators.anonymous_only_redirect(redirect_url='main'))
    def dispatch(self, request, *args, **kwargs):
        return super(LoginRequired, self).dispatch(request, *args, **kwargs)


class EmailRequired(EditablePageView):
    template_name = 'redirect.html'
    
    def get_context_data(self, **kwargs):
        render_context = 'email'
        return super(EmailRequired, self).get_context_data(
            validator_url=validator.get_validator_url(self.request), render_context=render_context,
            **kwargs
        )

    @method_decorator(view_decorators.yalogin_required_redirect(redirect_url=settings.LOGIN_URL))
    @method_decorator(view_decorators.no_email_only_redirect(redirect_url='main'))
    def dispatch(self, request, *args, **kwargs):
        render_context ='email'
        return super(EmailRequired, self).dispatch(request, *args, **kwargs)
