# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import logging

from django.core.urlresolvers import reverse
from django.db.models import Count
from django.db.models import Q
from django.http import Http404
from django.shortcuts import redirect
from django.utils.decorators import method_decorator
from django.views import generic as views
from django_yauth.decorators import yalogin_required

from app.views import mixins
from core import models
from core import forms
from core.utils import blackbox


__all__ = [
    'ReporterByName',
    'EditProfileView',
]

logger = logging.getLogger(__name__)


class ReporterByName(mixins.MenuMixin, views.DetailView):
    context_object_name = 'user'
    model = models.Reporter
    slug_field = 'uid'
    template_name = 'profile.html'

    def get_object(self, queryset=None):
        username = self.kwargs.get('username')
        try:
            uid = blackbox.external_userinfo_by_login(username).uid
        except blackbox.BlackboxUserError:
            raise Http404
        self.kwargs[self.slug_url_kwarg] = uid
        return super(ReporterByName, self).get_object(queryset)

    def get_queryset(self):
        user_uid = getattr(self.request.user, 'uid', None)
        queryset = models.Reporter.objects.filter(
            Q(show_on_hall=True) | Q(uid=user_uid))
        return queryset

    def get_context_data(self, **kwargs):
        return super(ReporterByName, self).get_context_data(
            awards=self.group_awards(), **kwargs)

    def group_awards(self):
        awards = self.object.userbadge_set.all()
        counted = awards.values('badge').annotate(count=Count('badge'))
        return list(self.mark_inactive_awards(awards, counted))

    def mark_inactive_awards(self, awards, counted):
        for count, badge_id in counted.values_list('count', 'badge'):
            last_award = awards.filter(badge_id=badge_id).last()
            is_inactive = last_award.deactivated is not None
            yield count, last_award.badge, is_inactive


class EditProfileView(mixins.MenuMixin, views.UpdateView):
    form_class = forms.ReporterForm
    template_name = 'profile_edit.html'

    def get_object(self, queryset=None):
        return self.request.user

    def get_success_url(self):
        return reverse('reporter-by-name', args=(self.request.yauser.login,))

    @method_decorator(yalogin_required)
    def dispatch(self, request, *args, **kwargs):
        return super(EditProfileView, self).dispatch(request, *args, **kwargs)
