# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import copy
import itertools
import os.path
import io
from operator import methodcaller

import xlwt

from django.conf import settings
from reportlab import platypus
from reportlab.lib import colors, enums, pagesizes, styles, units
from reportlab.pdfbase import pdfmetrics, ttfonts

from app.bblib.data import Winner
from core.models import Reward, Protocol, Responsible


one_cm = units.cm


class BaseReport(object):
    prefix = ''
    fileext = ''

    def __init__(self, start_date, end_date):
        self.start_date = start_date
        self.end_date = end_date

    @property
    def filename(self):
        return '{0.prefix}_{0.end_date:%d%m%Y}.{0.fileext}'.format(self)

    @staticmethod
    def _create_winner(reward):
        if reward.payment_currency is None:
            raise ValueError('payment_currency not specified for reward with id={}'.format(reward.id))
        reporter = reward.reporter
        datasync_values = reporter.new_payment_info.datasync_values
        winner = Winner(
            wid=str(reward.pk),
            reward_amount=reward.effective_payment_amount,
            reward_currency=reward.humanized_payment_currency,
            hall_name=reporter.username,
            country=datasync_values.get('country', '-'),
            contract=reporter.balance_contract_id,
            lname=datasync_values['lname'],
        )
        winner.name = ' '.join([datasync_values['lname'], datasync_values['fname']])
        winner.is_foreigner = (datasync_values.get('type') == 'ytph')
        return winner

    @classmethod
    def create_report(cls, protocol, foreigners=None):
        buffer = io.BytesIO()
        rewards = Reward.assigned.filter(protocol=protocol)
        start_date = protocol.start_date
        end_date = protocol.end_date
        winners = []
        instance = cls(start_date, end_date)
        for reward in rewards:
            winner = cls._create_winner(reward)
            if foreigners in [None, winner.is_foreigner]:
                winners.append(winner)
        winners.sort(key=methodcaller('ordering'))

        instance.generate(winners, buffer=buffer)
        buffer.seek(0)
        return buffer.read()

    def generate(self, *args, **kwargs):
        raise NotImplementedError


class FinancialReport(BaseReport):
    prefix = 'financial-report'
    fileext = 'xls'

    @classmethod
    def create_report(cls, protocol, **kwargs):
        return super(FinancialReport, cls).create_report(protocol)

    def get_width(self, num_characters):
        return int((1+num_characters) * 256)

    def generate(self, winners, buffer):
        book = xlwt.Workbook('utf8')
        font = xlwt.easyxf(
            'font: name Arial,colour_index black, bold off,italic off; align: wrap on, vert top, horiz left')
        
        borders = xlwt.Borders()
        borders.left = 1
        borders.top = 1
        borders.bottom = 1
        borders.right = 1
        font.borders = borders
        font_header = copy.deepcopy(font)
        alignment = xlwt.Alignment()
        alignment.horz = xlwt.Alignment.HORZ_CENTER
        font_header.alignment = alignment
        font_header.font.bold = True
        font_link = copy.deepcopy(font)
        font_link.font.underline = xlwt.Font.UNDERLINE_SINGLE
        font_link.font.colour_index = 4

        sheet = book.add_sheet('Все')
        header_row = (
            'Фамилия',
            'Сумма приза',
            'Валюта',
            'Резидент',
            'Договор',
        )

        for i, row in enumerate(header_row):
            sheet.write(0, i, row, font_header)
            sheet.col(i).width = self.get_width(15)
        sheet.col(0).width = self.get_width(30)
        winners = self.aggregate_payments(winners)

        for i, w in enumerate(winners, start=1):
            round_digits = 2
            row = (
                # A Фамилия
                w.lname,
                # B Сумма приза
                w.reward_amount,
                # C Валюта
                w.reward_currency,
                # D Резидент
                'Нет' if w.is_foreigner else 'Да',
                # D Договор
                w.contract,
            )
            for j, value in enumerate(row):
                sheet.write(i, j, value, font)

        book.save(buffer)
        return buffer

    def aggregate_payments(self, winners):
        aggregated = dict()
        for winner in winners:
            if winner.hall_name in aggregated:
                assert aggregated[winner.hall_name].reward_currency == winner.reward_currency
                aggregated[winner.hall_name].reward_amount += winner.reward_amount
            else:
                aggregated[winner.hall_name] = winner
        return list(aggregated.values())


class ProtocolReport(BaseReport):
    prefix = 'protocol'
    fileext = 'pdf'
    font_name = 'DejaVuSerif'

    @classmethod
    def create_report(cls, protocol, **kwargs):
        return super(ProtocolReport, cls).create_report(protocol, foreigners=True)

    @staticmethod
    def _month_name(index):
        return dict((
            (1, 'января'),
            (2, 'февраля'),
            (3, 'марта'),
            (4, 'апреля'),
            (5, 'мая'),
            (6, 'июня'),
            (7, 'июля'),
            (8, 'августа'),
            (9, 'сентября'),
            (10, 'октября'),
            (11, 'ноября'),
            (12, 'декабря'),
        )).get(index)

    def generate(self, winners, buffer):
        '''
        Genereate A4 (210×297) report
        '''
        self._register_fonts()
        story = list(itertools.chain(
            self._get_headers(),
            self._get_text(),
            self._get_winners(winners),
            self._get_rewards_text(),
            self._get_rewards_table(winners),
            self._get_signature_text()))

        doc = platypus.SimpleDocTemplate(
            buffer, pagesize = pagesizes.A4,
            leftMargin=one_cm, rightMargin=one_cm)
        doc.build(story)
        return buffer

    def _register_fonts(self):
        pdfmetrics.registerFont(ttfonts.TTFont(
            self.font_name,
            os.path.join(settings.FONTS_DIR, '%s.ttf' % self.font_name)))

    def _get_heading_style(self):
        styleH = styles.getSampleStyleSheet()['Heading1']
        styleH.fontName = self.font_name
        styleH.alignment = enums.TA_CENTER
        return styleH

    def _get_common_text_style(self):
        style = styles.getSampleStyleSheet()['Normal']
        style.fontName = self.font_name
        style.fontSize = 12
        style.leading = 16
        return style

    def _get_date_style(self):
        styleD = self._get_common_text_style()
        styleD.alignment = enums.TA_JUSTIFY
        return styleD

    def _get_text_style(self):
        styleN = self._get_common_text_style()
        styleN.firstLineIndent = 24
        styleN.alignment = enums.TA_JUSTIFY
        return styleN

    def _get_headers(self):
        yield platypus.Paragraph('ПРОТОКОЛ', self._get_heading_style())
        yield platypus.Paragraph(
            'о результатах  публичного конкурса «Охота за ошибками»',
            self._get_heading_style())
        yield platypus.Paragraph(
            'г. Москва «%s» %s %s года' % (
                self.end_date.strftime('%d'),
                self._month_name(self.end_date.month),
                self.end_date.strftime('%Y')),
            self._get_date_style())
        yield platypus.Spacer(one_cm, one_cm)

    def _get_text(self):
        try:
            chairman = Responsible.objects.filter(role=Responsible.ROLE_INFOSEC_CHAIRMAN).first().name_genitive
        except AttributeError:
            chairman = ''
        members = ''.join([
            ', члена комиссии {}'.format(name)
            for name in Responsible.objects.filter(
                role=Responsible.ROLE_INFOSEC_MEMBER).values_list('name_genitive', flat=True)
        ])
        text = '''Конкурсная комиссия, созданная в соответствии с Приказом №3-1/к от «18» марта 2020 года
        и Положением о публичном конкурсе «Охота за ошибками», в составе председателя комиссии 
        {}{} рассмотрела  поступившие в рамках проведения конкурса «Охота за ошибками» результаты 
        выполнения конкурсного задания на предмет соответствия критериям,  
        определенным в п. 7 Положения о публичном конкурсе «Охота за ошибками» и приняла следующие решения:'''.format(
            chairman, members
        )
        yield platypus.Paragraph(text, self._get_text_style())
        yield platypus.Spacer(one_cm, one_cm)
        text = '''1. Признать победителями конкурса «Охота за ошибками» 
        за период с %s по %s следующих участников:''' % (self.start_date.strftime('%d.%m.%Y'), self.end_date.strftime('%d.%m.%Y'))
        yield platypus.Paragraph(text, self._get_text_style())
        yield platypus.Spacer(one_cm, one_cm)

    def _get_winners(self, winners):
        names = set()
        for w in winners:
            names.add(w.condensed_name)
        names = sorted(names)
        items = [
            platypus.Paragraph(i, self._get_common_text_style()) for i in names]
        yield platypus.ListFlowable(
            items, bulletType='bullet', start='circle')
        yield platypus.Spacer(one_cm, one_cm)

    def _get_rewards_text(self):
        text = '''2. В соответствии с п.8 Положением о публичном конкурсе «Охота за ошибками» 
        выплатить победителям вознаграждение в следующем размере:'''
        yield platypus.Paragraph(text, self._get_text_style())
        yield platypus.Spacer(one_cm, one_cm)

    def _get_rewards_table(self, winners):
        data = [
            ['Победитель', 'Страна резидентства', 'Вознаграждение', 'Валюта', 'Идентификатор']
        ]

        for w in winners:
            data.append(
                [w.name, w.country, w.reward_amount, w.reward_currency,
                 'REWARD-%s' % w.wid]
            )

        table = platypus.Table(data)
        table.setStyle(platypus.TableStyle([
            ('BOX', (0, 0), (-1, -1), 0.25, colors.black),
            ('INNERGRID', (0, 0), (-1, -1), 0.25, colors.black),
            ('FONTNAME', (0, 0), (-1, -1), self.font_name),
            ('FONTSIZE', (0, 0), (-1, -1), 11),
            ('LEADING', (0, 0), (-1, -1), 14),
            ('ALIGN', (0, 0), (-1, -1), 'LEFT'),
            ('VALIGN', (0, 0), (-1, -1), 'MIDDLE'),
        ]))

        yield table
        yield platypus.Spacer(one_cm, one_cm)

    def _get_signature_text(self):
        for chairman in Responsible.objects.infosec_chairmen():
            yield platypus.Paragraph(
                'Председатель комиссии: ________________/ {}'.format(chairman.name),
                self._get_text_style()
            )
        for member in Responsible.objects.infosec_members():
            yield platypus.Paragraph(
                'Член комиссии:             ___________________/ {}'.format(member.name),
                self._get_text_style()
            )
