# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import decimal
from django import forms
from django.conf import settings
from django.core.exceptions import ValidationError
from django.utils.translation import ugettext_lazy as _

from core import models
from core.models import Reporter
from core.models import Reward
from core.utils import external_userinfo_by_login, BlackboxUserError


__all__ = [
    'RewardAPIForm',
    'RewardStatusForm',
    'RewardForm',
    'CreateRewardForm',
]


class RewardAPIForm(forms.ModelForm):
    # TODO(remedy) Someday we should properly replace department with product
    # BUGBOUNTY-339
    product = forms.ModelChoiceField(models.Product.objects.all(), required=False)
    department = forms.ModelChoiceField(models.Product.objects.all(), required=True)
    # TODO(remedy) We need reporter link but we get user_uid from OTRS
    # BUGBOUNTY-353
    user_uid = forms.IntegerField(required=True)
    reporter = forms.ModelChoiceField(models.Reporter.objects.all(), required=False)
    auth_token = forms.CharField(max_length=255)

    class Meta:
        model = Reward
        exclude = [
            'user_login', 'protocol', 'status', 'ticket_code', 'department',
            'balance_contract_attempts', 'balance_contract_created', 'balance_error_text',
        ]

    def __init__(self, *args, **kwargs):
        super(RewardAPIForm, self).__init__(*args, **kwargs)

    def clean_auth_token(self):
        auth_token = getattr(settings, 'API_AUTH_TOKEN')
        if self.cleaned_data['auth_token'] == auth_token:
            return auth_token
        raise forms.ValidationError('Token is invalid')

    def clean(self):
        self.cleaned_data['product'] = self.cleaned_data['department']
        user_uid = self.cleaned_data['user_uid']
        try:
            self.cleaned_data['reporter'] = models.Reporter.objects.get(uid=user_uid)
        except models.Reporter.DoesNotExist:
            raise ValidationError('reporter with uid %s is not registered' % user_uid)
        return self.cleaned_data 


class RewardStatusForm(forms.Form):
    status = forms.ChoiceField(choices=[(None, 'All')] + list(Reward.STATUS),
                               required=False, initial=None)


class BlackboxUserField(forms.CharField):

    def clean(self, value):
        user_login = super(BlackboxUserField, self).clean(value)
        try:
            user = external_userinfo_by_login(user_login)
        except BlackboxUserError:
            raise ValidationError('User with login %s is not found in blackbox' % user_login)
        try:
            reporter = Reporter.objects.get(uid=user.uid)
        except Reporter.DoesNotExist:
            raise ValidationError('User with login %s is not found among reporters' % user_login)
        return reporter


class ProductChoiceField(forms.ModelChoiceField):

    def label_from_instance(self, obj):
        return obj.title


MIN_PAYMENT_AMOUNT_THRESHOLD = decimal.Decimal('1.1')


def validate_amount_field(form, field_name):
    value = form.cleaned_data[field_name]
    if value < MIN_PAYMENT_AMOUNT_THRESHOLD:
        raise forms.ValidationError(_('reward.require_both_currencies_error'))
    return value


class RewardForm(forms.ModelForm):
    product = ProductChoiceField(
        queryset=models.Product.objects.all().order_by('title'))

    def clean_payment_amount_rur(self):
        return validate_amount_field(self, 'payment_amount_rur')

    def clean_payment_amount_usd(self):
        return validate_amount_field(self, 'payment_amount_usd')

    class Meta:
        model = Reward
        fields = [
            'vulnerability_type',
            'product',
            'payment_amount_rur',
            'payment_amount_usd',
            'points',
            'comment',
            'status',
        ]


class CreateRewardForm(forms.ModelForm):
    reporter = BlackboxUserField(required=True)
    staff_uid = forms.IntegerField(widget=forms.HiddenInput())
    product = ProductChoiceField(
        queryset=models.Product.objects.all().order_by('title'))

    def clean_payment_amount_rur(self):
        return validate_amount_field(self, 'payment_amount_rur')

    def clean_payment_amount_usd(self):
        return validate_amount_field(self, 'payment_amount_usd')

    def clean_startrek_ticket_code(self):
        if self.cleaned_data.get('startrek_ticket_code').startswith(settings.STARTREK_REWARD_QUEUE):
            return self.cleaned_data.get('startrek_ticket_code').replace('{}-'.format(settings.STARTREK_REWARD_QUEUE), '')
        return self.cleaned_data.get('startrek_ticket_code')
            
    class Meta(RewardForm.Meta):
        model = Reward
        fields = [
            'reporter',
            'startrek_ticket_code',
            'ticket_created',
            'vulnerability_type',
            'product',
            'payment_amount_rur',
            'payment_amount_usd',
            'points',
            'comment',
            'staff_uid',
            'staff_login',
        ]
