# -*- coding: utf-8 -*-

import json

from django_tanker import api
import yenv


DEFAULT_LANGUAGE = 'ru'
DEFAULT_DOWNLOAD_LANGUAGE = 'ru,en'
STATUS_APPROVED = 'approved'


class TankerException(Exception):
    pass


class MissingTranslationError(TankerException):
    pass


class TankerAdapter(api.Tanker):
    DEFAULT_FILE_FORMAT = 'tjson'

    def __init__(self, project_id, token, base_url=None):
        if base_url is None:
            if yenv.type == 'production':
                base_url = api.URLS['stable']
            else:
                base_url = api.URLS['testing']
        super(TankerAdapter, self).__init__(
            project_id=project_id, 
            base_url=base_url,
            token=token,
            dry_run=False,
            include_unapproved=False,
        )

    def create(self, keyset):
        return super(TankerAdapter, self).create(
            keyset=keyset.name,
            language=DEFAULT_LANGUAGE,
            path=keyset.data,
            key_not_language=False,
            file_format=self.DEFAULT_FILE_FORMAT,
        )

    def merge(self, keyset):
        return self.__upload(keyset, 'merge')

    def replace(self, keyset):
        return self.__upload(keyset, 'replace')

    def __upload(self, keyset, mode):
        return self.upload(
            keyset=keyset.name,
            language=DEFAULT_LANGUAGE,
            file_info=keyset.data,
            mode=mode,
            key_not_language=False,
            file_format=self.DEFAULT_FILE_FORMAT,
        )

    def download(self, keyset):
        return super(TankerAdapter, self).download(
            keyset=keyset.name,
            language=DEFAULT_DOWNLOAD_LANGUAGE,
            key_not_language=False,
            status=STATUS_APPROVED,
            file_format=self.DEFAULT_FILE_FORMAT,
        )


class TankerKeyset(object):
    KEYSET = None

    def __init__(self, slug, **kwargs):
        self.slug = slug
        self.name = '%s_%s' % (yenv.type, self.KEYSET)


class TankerUploadKeyset(TankerKeyset):
    CONTENT_TYPE = 'application/json'
    FILENAME = 'translation.json'

    def __init__(self, slug, **kwargs):
        super(TankerUploadKeyset, self).__init__(slug)
        keys = dict(self._init_keys(slug, **kwargs))
        self.data = self._init_translations(keys)

    def _init_keys(self, slug, **kwargs):
        for keyname, keyvalue in kwargs.items():
            key = '%s.%s' % (slug, keyname)
            value = {
                'translations': {
                    DEFAULT_LANGUAGE: {
                        'form': keyvalue,
                    },
                }
            }
            yield key, value

    def _init_translations(self, keys):
        return {
            'filename': self.FILENAME,
            'content_type': self.CONTENT_TYPE,
            'data': json.dumps({
                'keysets': {
                    self.name: {
                        'keys': keys,
                    },
                },
            }),
        }


class PageKeyset(TankerKeyset):
    KEYSET = 'pages'


class PageUploadKeyset(PageKeyset, TankerUploadKeyset):

    def __init__(self, slug, name, text):
        super(PageUploadKeyset, self).__init__(slug, name=name, text=text)


class PageKeysetTranslationAdapter(PageKeyset):
    KEYSET = 'pages'
    KEYMAP = {
        'name_ru': 'title_ru',
        'text_ru': 'text_ru',
        'name_en': 'title_en',
        'text_en': 'text_en',
    }

    def __init__(self, slug, content):
        super(PageKeysetTranslationAdapter, self).__init__(slug)
        self.content = json.loads(content)

    def get_translation_keys(self):
        keys = self.content['keysets'][self.name]['keys']
        for key, value in keys.items():
            translation_slug, translation_keyname = key.split('.', 1)
            if translation_slug != self.slug:
                continue
            translations = value['translations']
            for language, translation in translations.items():
                keyname = '%s_%s' % (translation_keyname, language)
                if translation['status'] != STATUS_APPROVED:
                    raise MissingTranslationError(keyname)
                else:
                    translation_value = translation['form']
                    yield self.KEYMAP[keyname], translation_value
