# -*- coding: utf-8 -*-
from __future__ import unicode_literals

from datetime import timedelta
import decimal
import logging
import os
import pytz

import yt.wrapper as yt
from celery import task
from django.conf import settings
from django.db import transaction
from django.utils import timezone

from core.models.reward import Reward
from core.models.user import NewPaymentInfo
from core.utils.balance import BalanceClient

__all__ = ['export_rewards_to_yt']

logger = logging.getLogger(__name__)

yt.config['proxy']['url'] = settings.YT_CLUSTER
yt.config['token'] = settings.YT_TOKEN


CURRENCY = {
    Reward.C_RUR: 'RUB',
    Reward.C_USD: 'USD',
}


MSK_TZ = pytz.timezone('Europe/Moscow')


@task
@transaction.atomic
def export_rewards_to_yt():
    today_midnight = timezone.now().astimezone(MSK_TZ).replace(hour=0, minute=0, second=0, microsecond=0)

    table_path = os.path.join(
        settings.YT_BALANCE_EXPORT_PATH,
        (today_midnight - timedelta(days=1)).strftime('%Y-%m-%d'),
    )

    # Не пересоздаем таблицу
    if yt.exists(table_path):
        return False
    rewards_qs = Reward.objects.filter(
        status=Reward.ST_PAYABLE,
        balance_contract_created__lte=today_midnight,
    ).all()
    rewards_to_export = []

    for reward in rewards_qs:
        rewards_to_export.append({
            'client_id': reward.reporter.balance_client_id,
            'money': unicode(reward.effective_payment_amount),
            'currency': CURRENCY[reward.payment_currency],
        })
        reward.status = Reward.ST_UPLOADED_TO_YT
        reward.save(update_fields=['status'])

    yt.create('table', table_path, ignore_existing=True)
    yt.write_table(table_path, rewards_to_export)



