# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import logging
from datetime import datetime, timedelta

from django.conf import settings
from celery import task

from app.utils.startrek import create_ticket, find_tickets

__all__ = ['report_failed_payments']

logger = logging.getLogger(__name__)


@task
def report_failed_payments():
    logger.info('Reporting failed payments')
    date_from = (datetime.now() - timedelta(weeks=1)).date()

    issues = find_tickets(filters={'queue': settings.STARTREK_OEBSSUPPORT_QUEUE,
                                   'author': settings.STARTREK_FAILED_PAYMENT_TICKET_AUTHOR,
                                   'tags': ['MONITORING'],
                                   'created': {'from': str(date_from)},})

    links = set()
    unique_records = set()
    for issue in issues:
        lines = [line.strip() for line in issue.description.replace('<br>', '\n').split('\n') if 'BUGBOUNTY' in line]
        if lines:
            links.add(issue.key)
            unique_records.update(lines)

    if not unique_records:
        logger.info('No failed payments found')
        return

    logger.info('Found {} issues with {} unique records'.format(len(links), len(unique_records)))

    column_names = ('БЮ', 'Тип отчёта', 'ID заг-ка', 'Договор', 'Плательщик', 'Период с', 'Период по', 'Дата платежа',
                    'К выплате всего', 'Сумма по ПСФ', 'Налог', 'Оплачено', 'Валюта', 'Результат')

    table_header = '#|\n|| ' + '|'.join(column_names) + '||\n'
    table_body = '\n'.join('||{}||'.format(row) for row in unique_records) + '\n|#'

    bountyfinance_issue = create_ticket(
        queue=settings.STARTREK_FAILED_PAYMENT_QUEUE,
        summary='Ошибки в выплатах призов ({} - {})'.format(date_from, datetime.now().date()),
        type=2,  # Задача?
        description=(
            'В тикет{} {} обнаружены сообщения об ошибках выплат в bugbounty.\n'
            'Список выплат:\n'
            '{}'
            '{}'
        ).format('ах' if len(links) > 1 else 'е', ', '.join(links), table_header, table_body)
    )
