# -*- coding: utf-8 -*-

import logging
import io
import locale

from celery import task
from django.utils import timezone
from django.conf import settings
from dateutil.relativedelta import relativedelta

from core.models import Protocol, Reward, Responsible
from app.bblib.reports import ProtocolReport, FinancialReport
from app.utils.startrek import create_ticket, get_startrek_ticket, get_latest_comment_attachment, comment_on_ticket_key
from app.utils.financial_report_audit import audit_protocol_rewards


__all__ = ['generate_protocol', 'create_kaznaops_ticket', 'create_document_ticket', 'check_document_ticket']

logger = logging.getLogger(__name__)

DOCUMENT_TICKET_DESCRIPTION ="""
Сверка состава выплат за {}.
Для проверки состава выплат создайте в этом тикете комментарий с прикрепленным файлом excel,
содержащим информацию о выплатах.
В течении нескольких минут в тикете должен появиться комментарий от robot-bugbounty с результатом проверки
(для просмотра новых комментариев понадобится обновить страницу).
Если комментарий от робота не появляется, напишите на tools@.
"""


@task
def create_kaznaops_ticket(protocol_id):
    locale.setlocale(locale.LC_TIME, 'ru_RU.UTF-8')
    protocol = Protocol.objects.get(pk=protocol_id)
    buffer = io.BytesIO()
    buffer.write(protocol.protocol_file)
    buffer.seek(0)
    buffer.name = 'report.pdf'
    infosec = Responsible.objects.filter(role__in=[Responsible.ROLE_INFOSEC_MEMBER, Responsible.ROLE_INFOSEC_CHAIRMAN])
    summonees = list(infosec.values_list('staff_username', flat=True))
    month = timezone.datetime.strftime(protocol.start_date, '%B').decode('utf-8').lower().encode('utf-8')
    issue = create_ticket(
        queue=settings.STARTREK_KAZNAOPERATIONS_QUEUE,
        summary='Охота за ошибками: выплаты нерезидентам за {}'.format(month),
        description='Для проведения выплат по результатам конкурса '
                    '"Охота за ошибками" требуется подписать протокол (во вложении)',
        attachment=buffer,
        summonees=summonees,
    )
    protocol.ticket_kaznaops = issue.key
    protocol.save(update_fields=['ticket_kaznaops'])


@task
def create_document_ticket(protocol_id):
    locale.setlocale(locale.LC_TIME, 'ru_RU.UTF-8')
    protocol = Protocol.objects.get(pk=protocol_id)
    supervisors = Responsible.objects.filter(role=Responsible.ROLE_SUPERVISOR)
    summonees = list(supervisors.values_list('staff_username', flat=True))
    datestring = timezone.datetime.strftime(protocol.start_date, '%B %Y')
    issue = create_ticket(
        queue=settings.STARTREK_DOCUMENT_QUEUE,
        summary='Проверка - {} - Охота за ошибками'.format(datestring),
        description=DOCUMENT_TICKET_DESCRIPTION.format(datestring),
        summonees=summonees,
        tags=['bugbounty'],
    )
    protocol.ticket_document = issue.key
    protocol.save(update_fields=['ticket_document'])


@task
def check_document_ticket(protocol_id):
    protocol = Protocol.objects.get(pk=protocol_id)
    external_content = get_latest_comment_attachment(protocol.ticket_document)
    if not external_content:
        return
    try:
        missing, extra = audit_protocol_rewards(protocol, external_content)
    except Exception:
        comment_on_ticket_key(protocol.ticket_document, u'Не получилось прочитать документ')
        return

    comment_text = []
    summonees = None
    if not (missing or extra):
        comment_text = [u'Состав выплат в порядке.']
    else:
        responsibles = Responsible.objects.filter(
            role__in=[Responsible.ROLE_SUPERVISOR, Responsible.ROLE_BUGBOUNTY_RESPONSIBLE])
        summonees = list(responsibles.values_list('staff_username', flat=True))
    if missing:
        comment_text.append(u'В приведенном документе не хватает следующих записей:')
        comment_text.extend([u' '.join(record) for record in missing])
    if extra:
        if missing:
            comment_text.append(u'')
        comment_text.append(u'Следующие записи в приведенном документе лишние:')
        comment_text.extend([u' '.join(record) for record in extra])
    comment_on_ticket_key(protocol.ticket_document, u'\n'.join(comment_text), summonees=summonees)


@task
def generate_protocol():
    first_day_of_last_month = (timezone.now() - relativedelta(months=1)).replace(
        day=1, hour=0, minute=0, second=0, microsecond=0)
    first_day_of_current_month = timezone.now().replace(day=1, hour=0, minute=0, second=0, microsecond=0)
    rewards = Reward.objects.filter(
        status=Reward.ST_UPLOADED_TO_YT,
        balance_contract_created__range=[first_day_of_last_month, first_day_of_current_month],
        protocol=None,
    )
    if not rewards.exists():
        logger.info('No new rewards found. Protocol not created.')
        return


    protocol = Protocol.objects.create(
        staff_uid=None,
        staff_login=None,
        start_date=first_day_of_last_month,
    )
    rewards.update(protocol=protocol)
    protocol.protocol_file = ProtocolReport.create_report(protocol)
    protocol.financial_report_file = FinancialReport.create_report(protocol)
    protocol.save(update_fields=['protocol_file', 'financial_report_file'])

    logger.info('Protocol for %s rewards created.' % rewards.count())
