# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import io
from ids.registry import registry
from startrek_client import Startrek

from django.conf import settings


def get_issues_repository():
    issues_repository = registry.get_repository(
        'startrek2',
        'issues',
        user_agent=settings.BUGBOUNTY_ROBOT_USERNAME,
        oauth_token=settings.STARTREK_TOKEN,
        timeout=settings.STARTREK_TIMEOUT,
    )
    return issues_repository


def get_startrek_ticket(ticket_key):
    repo = get_issues_repository()
    issue = repo.get_one({'id': ticket_key})
    return issue


def get_latest_comment_attachment(ticket_key):
    client = Startrek(
        useragent=settings.BUGBOUNTY_ROBOT_USERNAME,
        token=settings.STARTREK_TOKEN,
        base_url=settings.STARTREK_BACKEND_BASE_URL,
    )
    issue = client.issues[ticket_key]
    last_comment = sorted(list(issue.comments.get_all()), key=lambda x: x.createdAt)[-1]
    comment_attachments = [attach for attach in issue.attachments.get_all() if
                           getattr(attach, 'commentId', None) == last_comment.longId]
    if not comment_attachments:
        return None
    attachment = comment_attachments[0]
    buffer = io.BytesIO()
    for chunk in client._connection.stream(path=attachment.content):
        buffer.write(chunk)
    buffer.seek(0)
    return buffer.read()


def comment_on_ticket_key(ticket_key, text, summonees=None):
    ticket = get_startrek_ticket(ticket_key)
    kwargs = {'text': text}
    if summonees:
        kwargs['summonees'] = summonees
    ticket.comments.create(**kwargs)


def notify_about_payment_issues(reward):
    if not reward.startrek_ticket_code:
        raise ValueError('Could not comment on non-existent issue for reward={}'.format(reward.pk))
    errors = reward.balance_error_text
    text = '''
Произошла ошибка при заведении платёжного договора в Балансе:

%%{}%%

Если ошибка связана с неправильным заполнением платёжных данных, требуется связаться с пользователем для их исправления.
Иначе призовите в тикет разработчика BugBounty для уточнения причины ошибки.
    '''.format(errors).strip('\n')
    ticket_key = '{}-{}'.format(settings.STARTREK_REWARD_QUEUE, reward.startrek_ticket_code)
    comment_on_ticket_key(ticket_key, text)


def create_ticket(queue, summary, description, attachment=None, summonees=None, type=None, tags=None):
    repository = get_issues_repository()
    client = repository.client

    kwargs = dict(
        queue=queue,
        summary=summary,
        description=description
    )

    if tags:
        kwargs['tags'] = tags

    if type is not None:
        kwargs['type'] = type

    issue = client.issues.create(**kwargs)

    if attachment is not None:
        issue.attachments.create(attachment)

    if summonees:
        issue.comments.create(summonees=summonees)

    return issue


def find_tickets(filters):
    repository = get_issues_repository()
    client = repository.client
    issues = client.issues.find(filter=filters)
    return issues
