# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import logging

from django.views.decorators.csrf import csrf_exempt
from django.views.generic.base import View
from django.http import response
from django.utils.decorators import method_decorator

from app.forms import RewardAPIForm
from core import models


logger = logging.getLogger(__name__)


class RewardAPI(View):

    def post(self, request):
        form = RewardAPIForm(request.POST)
        if form.is_valid():
            instance = form.save(commit=False)
            user_uid = instance.user_uid
            try:
                instance.reporter = models.Reporter.objects.get(uid=user_uid)
            except models.Reporter.DoesNotExist:
                logger.error('Reporter not found', extra={'request': request})
                return response.JsonResponse(
                    dict(status='reporter_not_found', user_uid=user_uid),
                    status=400)
            instance.apply_badge_bonus()
            instance.save()
            return response.JsonResponse(dict(id=instance.id, status='ok'))
        return response.JsonResponse(
            dict(status='validation_error', errors=form.errors), status=400)

    def get(self, request, reward_id=None):
        objects = models.Reward.objects
        if reward_id is not None:
            objects = objects.filter(id=reward_id)
        data = list(objects.values('id', 'status'))
        return response.JsonResponse(data, safe=False)

    @method_decorator(csrf_exempt)
    def dispatch(self, *args, **kwargs):
        return super(RewardAPI, self).dispatch(*args, **kwargs)


class ConstantsList(View):
    """Abstract view class for constants' API."""

    def get(self, request):
        return response.JsonResponse(list(self.get_constants()), safe=False)


class DepartmentList(ConstantsList):

    def get_constants(self):
        for pk, title in models.Product.objects.values_list('id', 'title').order_by('title'):
            yield dict(id=pk, name=title)


class CurrencyList(ConstantsList):

    def get_constants(self):
        for currency_id, currency_name_proxy in models.Reward.CURRENCY:
            currency_name = '%s' % currency_name_proxy
            yield dict(id=currency_id, name=currency_name)


class Ping(View):

    def get(self, request):
        from django.db import connection
        from django.http import HttpResponse
        from datetime import datetime
        import smtplib
        # check MySql status
        try:
            cursor = connection.cursor()
            cursor.execute("SELECT 1")
            cursor.fetchone()
        except Exception as e:
            return HttpResponse("2;BD error: %s" % str(e), status=200)
        # check SMTP status
        try:
            smtplib.SMTP('localhost').ehlo()
        except Exception as e:
            return HttpResponse("2;SMTP error: %s" % str(e), status=200)
        return HttpResponse('0;OK', status=200)
