from app.forms import BlackboxUserForm
from app.forms import HallOfFamePublishForm
from app.forms import RemoveHallOfFameMemberForm
from django.contrib import messages
from django.contrib.auth.decorators import permission_required
from django.core.urlresolvers import reverse
from django.http import HttpResponseRedirect
from django.utils.decorators import method_decorator
from django.utils.translation import gettext_lazy as _
from django.views.generic import FormView
from django.views.generic import ListView
from django_yauth.decorators import yalogin_required

from core.models import HallOfFameGroup
from core.models import Item

__all__ = [
    'HallOfFameList',
    'AddHallOfFameMember',
    'PublishHallOfFame',
    'EditHallOfFame',
]


class HallOfFameList(ListView):
    context_object_name = 'items'
    paginate_by = 5
    template_name = 'hall_of_fame.html'

    def get_queryset(self):
        self.year = self.kwargs['year']
        self.month = self.kwargs['month']
        return Item.objects.filter(
            list__year=self.year,
            list__month=self.month).order_by('-id')

    def get_context_data(self):
        hall_of_fame = self._get_hall_of_fame()
        return super(HallOfFameList, self).get_context_data(
            year=self.year, month=self.month, hall_of_fame=hall_of_fame)

    def _get_hall_of_fame(self):
        try:
            hall_of_fame = HallOfFameGroup.objects.get(
                year=self.year, month=self.month)
        except HallOfFameGroup.DoesNotExist:
            hall_of_fame = None
        return hall_of_fame

    @method_decorator(yalogin_required)
    @method_decorator(permission_required('core.change_item', raise_exception=True))
    def dispatch(self, request, *args, **kwargs):
        return super(HallOfFameList, self).dispatch(request, *args, **kwargs)


class HallOfFameEditMixin(object):

    def get(self, request, *args, **kwargs):
        return HttpResponseRedirect(self.get_success_url())

    def _get_hall_of_fame(self):
        year = self._get_year()
        month = self._get_month()
        hall_of_fame, _ = HallOfFameGroup.objects.get_or_create(year=year, month=month)
        return hall_of_fame

    def _get_year(self):
        return self.kwargs['year']

    def _get_month(self):
        return self.kwargs['month']
    
    def get_success_url(self):
        year = self._get_year()
        month = self._get_month()
        return reverse('hall_of_fame:month-list',
                       kwargs={'year': year, 'month': month})

    @method_decorator(yalogin_required)
    @method_decorator(permission_required('core.add_list', raise_exception=True))
    @method_decorator(permission_required('core.change_list', raise_exception=True))
    def dispatch(self, request, *args, **kwargs):
        return super(HallOfFameEditMixin, self).dispatch(
            request, *args, **kwargs)


class AddHallOfFameMember(HallOfFameEditMixin, FormView):
    form_class = BlackboxUserForm

    def form_valid(self, form):
        reporter = form.cleaned_data['username']
        hall_of_fame = self._get_hall_of_fame()
        hall_of_fame.add_member(reporter)
        messages.success(self.request,
                         _('%s was added to hall of fame') % reporter.login)
        return super(AddHallOfFameMember, self).form_valid(form)

    def form_invalid(self, form):
        for error in form.non_field_errors():
            messages.error(self.request, error)
        return HttpResponseRedirect(self.get_success_url())


class PublishHallOfFame(HallOfFameEditMixin, FormView):
    form_class = HallOfFamePublishForm

    def form_valid(self, form):
        hall_of_fame = self._get_hall_of_fame()
        hall_of_fame.is_published = form.cleaned_data['is_published']
        hall_of_fame.save()
        messages.success(self.request, _('Successfully published'))
        return super(PublishHallOfFame, self).form_valid(form)

    def form_invalid(self, form):
        messages.error(self.request, _('Failed to publish'))
        return HttpResponseRedirect(reverse('hall_of_fame:list'))


class EditHallOfFame(HallOfFameEditMixin, FormView):
    form_class = RemoveHallOfFameMemberForm

    def form_valid(self, form):
        reporter_id = form.cleaned_data['reporter_id']
        year = self._get_year()
        month = self._get_month()
        Item.objects.filter(
            list__year=year, list__month=month, user=reporter_id).delete()
        messages.success(self.request, _('User was removed from hall of fame'))
        return super(EditHallOfFame, self).form_valid(form)

    def form_invalid(self, form):
        messages.error(self.request, _('Failed to remove'))
        return HttpResponseRedirect(self.get_success_url())
