# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import logging

from app import forms
from app.utils import otrs
from app.views import mixins
from django.conf import settings
from django.contrib import messages
from django.core.exceptions import ValidationError
from django.core.urlresolvers import reverse_lazy
from django.db import transaction
from django.shortcuts import get_object_or_404
from django.shortcuts import redirect
from django.utils.translation import gettext_lazy as _
from django.views import generic as views
from requests import exceptions

from core import models
from core.utils import external_userinfo_by_uid, BlackboxUserError

__all__ = ['RewardList', 'RewardView', 'CreateReward', 'BrowserRewardList']

logger = logging.getLogger(__name__)


class BaseRewardList(views.ListView):
    context_object_name = 'rewards'

    def get_context_data(self, **kwargs):
        context = super(BaseRewardList, self).get_context_data(
            form=self.form,
            status=self.status,
            protocol=self.get_protocol(),
            **kwargs)
        rewards = context[self.context_object_name]
        object_list = list(self.get_context_object(rewards))
        context[self.context_object_name] = object_list
        return context

    def get_context_object(self, rewards):
        uids = set(self.get_reward_reporter_uids(rewards))
        users = {}
        for uid in uids:
            try:
                user = external_userinfo_by_uid(uid)
            except BlackboxUserError:
                users[uid] = None
            else:
                users[user.uid] = user.username
        for reward in rewards:
            username = None
            if reward.reporter:
                username = users[reward.reporter.uid]
            yield (reward, username)

    def get_reward_reporter_uids(self, rewards):
        for reward in rewards:
            if reward.reporter:
                yield reward.reporter.uid

    def get_queryset(self):
        queryset = super(BaseRewardList, self).get_queryset()
        status = self.get_status()
        if status is not None:
            queryset = queryset.filter(status=status)
        protocol = self.get_protocol()
        if protocol is not None:
            queryset = queryset.filter(protocol_id=protocol)
        return queryset.select_related('reporter')

    def get_status(self):
        self.status = None
        self.form = forms.RewardStatusForm(self.request.GET)
        if self.form.is_valid():
            try:
                self.status = int(self.form.cleaned_data['status'])
            except ValueError:
                pass
        return self.status

    def get_protocol(self):
        try:
            protocol = int(self.request.GET.get('protocol'))
        except (TypeError, ValueError):
            protocol = None
        return protocol


class RewardList(mixins.SuperuserRequiredMixin, BaseRewardList):
    model = models.Reward
    paginate_by = 10
    template_name = 'rewards.html'


class BrowserRewardList(mixins.GroupRequiredMixin, BaseRewardList):
    model = models.Reward
    paginate_by = 10
    template_name = 'rewards-browser.html'
    group_required = settings.BROWSER_MANAGERS_GROUP

    def get_queryset(self):
        queryset = super(BrowserRewardList, self).get_queryset()
        queryset = queryset.filter(
            product__financial_unit__acronym=settings.BROWSER_FINANCIAL_UNIT)
        queryset = self.sort_queryset(queryset)
        return queryset.select_related('reporter')

    def sort_queryset(self, queryset):
        criterion = self._get_criterion()
        reverse = self._get_reverse()
        if reverse:
            criterion = '-' + criterion
        if criterion:
            queryset = queryset.order_by(criterion)
        return queryset

    def get_context_data(self, **kwargs):
        context = super(BrowserRewardList, self).get_context_data(**kwargs)
        criterion = self._get_criterion()
        reverse = self._get_reverse()
        if criterion and not reverse:
            context_variable_name = 'reverse_%s' % criterion
            context[context_variable_name] = True
        return context

    def _get_criterion(self):
        return self.request.GET.get('sort')

    def _get_reverse(self):
        return self.request.GET.get('reverse')


class RewardView(mixins.SuperuserRequiredMixin, views.TemplateView):
    template_name = 'reward-edit.html'

    def get(self, request, reward_id):
        reward = get_object_or_404(models.Reward, pk=reward_id)
        reporter = reward.reporter
        form = forms.RewardForm(instance=reward)
        return self.render_to_response(locals())

    def post(self, request, reward_id):
        reward = get_object_or_404(models.Reward, pk=reward_id)
        form = forms.RewardForm(request.POST, instance=reward)
        if form.is_valid():
            form.save()
            messages.success(request, _('Reward was successfully updated'))
            return redirect('reward:list')
        return self.render_to_response(locals())

    # TODO: grants.
    @transaction.atomic
    def delete(self, request, reward_id):
        updated = models.Reward.objects.filter(
            pk=reward_id,
            status=models.Reward.ST_PAYABLE,
        ).update(
            status=models.Reward.ST_DISCARDED,
        )
        return redirect('reward:list')


class CreateReward(mixins.SuperuserRequiredMixin, views.CreateView):
    form_class = forms.CreateRewardForm
    model = models.Reward
    template_name = 'reward-create.html'
    success_url = reverse_lazy('reward:list')

    def form_valid(self, form):
        instance = form.save(commit=False)
        instance.apply_badge_bonus()
        messages.success(self.request, _('Reward successfully created'))
        return super(CreateReward, self).form_valid(form)


class UserRewardList(mixins.SuperuserRequiredMixin, BaseRewardList):
    model = models.Reward
    paginate_by = 10
    template_name = 'rewards.html'
    reporter = None

    def get_queryset(self):
        queryset = super(UserRewardList, self).get_queryset()
        if self.reporter:
            queryset = queryset.filter(reporter=self.reporter)
        return queryset.select_related('reporter')

    def get(self, request, *args, **kwargs):
        self.reporter = get_object_or_404(models.Reporter, pk=kwargs.get('pk', -1))
        return super(UserRewardList, self).get(request, *args, **kwargs)
