# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import six

from django.conf import settings
from django.contrib import messages
from django.core.urlresolvers import reverse
from django.http import Http404, HttpResponse
from django.shortcuts import get_object_or_404
from django.shortcuts import redirect
from django.utils.translation import gettext_lazy as _
from django.views import generic as views
from django.views.generic.edit import BaseFormView

from app import forms as app_forms
from app.views import mixins
from core import forms as core_forms
from core import models
from core.utils import mail


class UserProfileList(mixins.GroupRequiredMixin, views.ListView):
    context_object_name = 'users'
    form_class = app_forms.BlackboxUserForm
    group_required = settings.FINANCIAL_SUPPORT_GROUP
    paginate_by = 5
    template_name = 'profiles.html'

    def get_queryset(self):
        form = self.get_form()
        if form.is_valid():
            users = [form.cleaned_data['username']]
        else:
            users = models.Reporter.objects.all()
        return users

    def get_form(self):
        return self.form_class(self.request.GET)

    def get_context_data(self):
        return super(UserProfileList, self).get_context_data(form=self.get_form())


class UpdateUserProfile(mixins.SuperuserRequiredMixin, views.UpdateView):
    form_class = core_forms.ReporterForm
    model = models.Reporter
    template_name = "profile_edit.html"

    def get_context_data(self, **kwargs):
        payment_info_url = None
        payment_info = models.NewPaymentInfo.objects.filter(reporter=self.object).first()
        if payment_info:
            payment_info_url = reverse(
                'user_profile:edit_payment',
                kwargs={'pk': self.object.pk}
            )
        user_rewards_url = reverse(
            'user_profile:filter_rewards',
            kwargs={'pk': self.object.pk}
        )
        return super(UpdateUserProfile, self).get_context_data(
            payment_info_url=payment_info_url,
            user_rewards_url=user_rewards_url,
            **kwargs
        )

    def form_valid(self, form):
        messages.success(self.request, _('User successfully updated'))
        return super(UpdateUserProfile, self).form_valid(form)


class EditUserProfile(UpdateUserProfile):
    """Legacy user profile view."""


class UserProfileByUid(UpdateUserProfile):
    """
    Get user profile by uid.

    Links to this view are included in OTRS tickets.
    """
    slug_field = 'uid'
    slug_url_kwarg = 'uid'


class EditUserPaymentInfo(mixins.GroupRequiredMixin, views.TemplateView):
    group_required = settings.FINANCIAL_SUPPORT_GROUP
    template_name = 'payment_edit.html'

    def get_initial(self, instance):
        is_russian_resident = self.request.LANGUAGE_CODE == 'ru'
        bank_type = '0'  # не выбрано
        if instance is not None:
            # is_russian_resident – или тип "резидент РФ", или если пока ещё ничего не известно
            is_russian_resident = instance.datasync_values.get('type') in (models.NewPaymentInfo.RESIDENT_TYPE, None)
            bank_type = instance.datasync_values.get('bank-type', '0')

        initial = {}
        for key, value in instance.datasync_values.items():
            if key in ['id', 'type', 'document']:
                continue
            form_key = key.replace('-', '_')
            initial[form_key] = value

        initial.update({
            'is_russian_resident': is_russian_resident,
            'bank_type': bank_type,
        })
        return initial

    def get(self, request, pk):
        reporter = get_object_or_404(models.Reporter, pk=pk)
        try:
            payment_info = models.NewPaymentInfo.objects.get(reporter=reporter)
        except models.NewPaymentInfo.DoesNotExist:
            payment_info = models.NewPaymentInfo(reporter=reporter)
        form = core_forms.NewPaymentInfoForm(instance=payment_info, initial=self.get_initial(payment_info))
        document_name = document_url = ''
        document = payment_info.datasync_values.get('document')
        if document:
            document_name = document.name
            document_url = reverse('user_profile:download_id_document', kwargs={'pk': reporter.pk})
        if payment_info.document_uploaded_at:
            document_uploaded_at = payment_info.document_uploaded_at.strftime('%Y-%m-%d %H:%M UTC')
        else:
            document_uploaded_at = ''
        context = {
            'reporter': reporter,
            'payment_info': payment_info,
            'form': form,
            'document_uploaded_at': document_uploaded_at,
            'document_name': document_name,
            'document_url': document_url,
        }
        return self.render_to_response(context)

    def post(self, request, pk):
        reporter = get_object_or_404(models.Reporter, pk=pk)
        try:
            instance = models.NewPaymentInfo.objects.get(reporter=reporter)
        except models.NewPaymentInfo.DoesNotExist:
            instance = models.NewPaymentInfo(reporter=reporter)
        form = core_forms.NewPaymentInfoForm(
            data=request.POST,
            files=request.FILES,
            instance=instance,
            initial=self.get_initial(instance),
        )
        if form.is_valid():
            form.save()
            messages.success(
                request,
                _('User payment info successfully updated'),
            )
        else:
            if 'document' in form.errors:
                messages.error(request, _('payment_info.invalid_document'))
            if 'document' not in form.errors or {'document'} < set(form.errors):
                messages.error(request, _('payment_info.invalid_form'))

        context = {
            'reporter': reporter,
            'instance': instance,
            'form': form
        }

        return self.render_to_response(context)


class DownloadUserDocument(mixins.GroupRequiredMixin, views.View):
    group_required = settings.FINANCIAL_SUPPORT_GROUP

    def get(self, request, pk):
        payment_info = get_object_or_404(models.NewPaymentInfo, reporter__pk=pk)
        document = payment_info.datasync_values.get('document')
        if not document:
            raise Http404('The reporter does not have attached document.')

        response = HttpResponse(document.read())
        response['content_type'] = 'application/octet-stream'
        response['Content-Disposition'] = 'attachment;filename={}'.format(document.name)
        return response


class DeleteUserBadge(mixins.SuperuserRequiredMixin, views.View):

    def post(self, request, pk):
        form = app_forms.UserBadgeForm(request.POST)
        if form.is_valid():
            data = form.cleaned_data
            try:
                user_badge = models.UserBadge.active.get(**data)
                user_badge.deactivate()
                messages.success(request, _('Badge was successfully deactivated'))
            except models.UserBadge.DoesNotExist:
                messages.error(request, _('Badge does not exist'))
        else:
            messages.error(request, _('Failed to deactivate badge'))
        return redirect('user_profile:edit', pk)


class AddUserBadge(mixins.SuperuserRequiredMixin, BaseFormView):
    form_class = app_forms.BlackboxUserForm

    def form_valid(self, form):
        self.create_user_badge(form)
        return super(AddUserBadge, self).form_valid(form)

    def create_user_badge(self, form):
        badge = get_object_or_404(models.Badge, pk=self.get_badge_pk())
        reporter = form.cleaned_data['username']
        user_badge = models.UserBadge.objects.create(user=reporter, badge=badge)
        try:
            message = mail.BadgeMessage(user_badge)
        except mail.BadgeMessageError:
            messages.error(self.request,
                           _('MailTemplate for %s does not exist') % badge.template_code)
        else:
            message.send()
        messages.success(self.request, _('Badge was successfully assigned'))

    def form_invalid(self, form):
        messages.error(self.request, _('User does not exist'))
        return redirect('badge', self.get_badge_pk())

    def get_success_url(self):
        return reverse('badge', args=(self.get_badge_pk(),))

    def get_badge(self):
        return models.Badge.objects.get(pk=self.get_badge_pk())

    def get_badge_pk(self):
        return self.kwargs.get('pk')
