from django.shortcuts import get_object_or_404
from django.views.generic.base import TemplateView, View
from django.http import HttpResponse, HttpResponseRedirect
from django.utils.decorators import method_decorator
from django.utils.translation import gettext_lazy as _
from django.core.urlresolvers import reverse

import json
from app.forms import VulnerabilityForm

from core.models import Vulnerability
from django.contrib import messages
from django.contrib.auth.decorators import permission_required

from django_yauth.decorators import yalogin_required


class VulnerabilityView(TemplateView):

    template_name = 'vulnerability_list.html'

    def get_context_data(self, **kwargs):
        context = super(VulnerabilityView, self).get_context_data(**kwargs)
        context['vulnerabilities'] = Vulnerability.objects.all().order_by('name')
        return context

    def get(self, request, *args, **kwargs):
        context = self.get_context_data(**kwargs)
        if request.GET.get('json', False):
            vulnerabilities = list(context['vulnerabilities'].values())
            response = HttpResponse(content_type='application/json')
            json.dump(
                dict(vulnerabilities=vulnerabilities),
                response,
                indent=4,
            )
            return response
        return self.render_to_response(context)

    def dispatch(self, request, *args, **kwargs):
        dispatcher = super(VulnerabilityView, self).dispatch
        if not request.GET.get('json', False):
            dispatcher = yalogin_required(dispatcher)
        return dispatcher(request, *args, **kwargs)


class VulnerabilityDetailsView(View):
    @staticmethod
    def render_to_json(serializable, status_code=200):
        response = HttpResponse(content_type='application/json')
        response.status_code = status_code
        json.dump(serializable, response, indent=4)
        return response

    def post(self, request, pk=None):
        instance = None
        if pk is not None:
            instance = get_object_or_404(Vulnerability, pk=pk)
        form = VulnerabilityForm(request.POST, instance=instance)
        response = dict()
        if form.is_valid():
            instance = form.save()
            status_code = 200
            response['id'] = instance.pk
        else:
            status_code = 400
            response['errors'] = form.errors
        return self.render_to_json(response, status_code=status_code)

    @method_decorator(yalogin_required)
    @method_decorator(permission_required('core.change_vulnerability', raise_exception=True))
    def dispatch(self, request, *args, **kwargs):
        return super(
            VulnerabilityDetailsView,
            self,
        ).dispatch(request, *args, **kwargs)


class VulnerabilityDelete(TemplateView):

    template_name = 'vulnerability_list.html'

    def post(self, request):
        try:
            pk = request.POST['pk']
            vulnerability = get_object_or_404(Vulnerability, pk=pk)
            vulnerability.delete()
        except KeyError:
            messages.error(request, _('Vulnerability id not passed'))
        return HttpResponseRedirect(reverse('vulnerability:list'))

    @method_decorator(yalogin_required)
    @method_decorator(permission_required('core.delete_vulnerability', raise_exception=True))
    def dispatch(self, request, *args, **kwargs):
        return super(
            VulnerabilityDelete,
            self,
        ).dispatch(request, *args, **kwargs)
