# coding: utf-8
from __future__ import unicode_literals

import json
import logging
import yenv

from django import http
from django.utils.datastructures import MultiValueDict
from django.views.decorators.http import require_GET, require_POST

from cab import widgets
from cab.core.views import BaseView
from cab.utils import views
from cab.utils.helpers import timeit
from cab.utils.oebs import get_stock_options, push_plan, get_plan

log = logging.getLogger(__name__)


class WidgetList(BaseView):

    def get(self, request):
        return [
            {
                'id': id,
                'data_url': request.get_raw_uri() + id + '/'
            }
            for id in (
                'actions',
                'review',
                'persons',
                'info',
            )
        ]


class WidgetDetail(BaseView):

    @views.responding_json
    def dispatch(self, request, *args, **kwargs):

        widget_id = kwargs.pop('widget_id')
        widget = self.get_widget_or_404(widget_id=widget_id)

        endpoint_name = kwargs.pop('endpoint')
        handler = widget.get_handler(
            endpoint_name=endpoint_name,
            request=request,
        )

        # data is django.QueryDict
        # for better use of django forms as validators
        data = request.GET.copy()
        data.update(kwargs)

        if request.body:
            if 'application/json' in request.META['CONTENT_TYPE']:
                # a lot of hacks with querydicts, maybe reconsider
                # simple update makes QueryDict({'x': [['a', 'b']]})
                # but instantiating MultiValueDict
                # makes MultiValueDict({'x': ['a', 'b']})
                post_data = json.load(request)
                post_data = MultiValueDict({
                    key: value if isinstance(value, list) else [value] for
                    key, value in post_data.items()
                })
            else:
                post_data = request.POST
            log.debug('post_data %s', post_data)
            data.update(post_data)

        validator_cls = handler.validator_cls or getattr(handler, 'Validator', None)
        if validator_cls:
            validator = validator_cls(data=data)

            if not validator.is_valid():
                return self.error(validator.errors)
            else:
                data = validator.cleaned_data

        handler_method = timeit(handler.handle)
        return handler_method(data=data)

    @staticmethod
    def get_widget_or_404(widget_id):
        widget_cls = widgets.registry.get(widget_id)
        if widget_cls is None:
            raise http.Http404
        return widget_cls


def get_login(request, login):
    if yenv.type in ('testing', 'unstable'):
        return login or request.auth.login

    return request.auth.login


@require_GET
@views.responding_json
def stock_options(request, login=None):
    need_all_companies = request.GET.get('all_companies', False)  # default False for 10b5 form
    return get_stock_options(get_login(request, login), need_all_companies)


@require_POST
@views.responding_json
def push_plan_insider(request):
    try:
        plan_data = json.loads(request.body)
    except ValueError:
        return {'error': 'invalid format'}, 400

    login = request.auth.login
    return push_plan(login, plan_data)


@require_GET
@views.responding_json
def get_plan_insider(request):
    login = request.auth.login
    return get_plan([login])
