# coding: utf-8

from __future__ import unicode_literals

from cab.utils import env
from cab.utils import http
from cab.utils import helpers
from cab.utils import datetimes


URL = env.choose({
    'production': 'https://staff.yandex-team.ru',
    'testing': 'https://staff.test.yandex-team.ru',
    # 'development': 'https://port-9100.staff01.dev.yandex-team.ru/gap-api',
})

RESOURCES = {
    'need_for_attention': URL + '/gap-api/api/need_for_attention/',
    'confirm_api': URL + '/gap-api/api/gap_confirm/{gap_id}/',
    'gaps': URL + '/gap-api/api/export_gaps/',
    'confirm_web': URL + '/gap/{gap_id}/',
}


@helpers.timeit
def get_vacations(auth):
    session = http.get_session(auth)
    raw_data = session.get(RESOURCES['need_for_attention']).json()
    return map(Gap, raw_data['unconfirmed_vacations'])


@helpers.timeit
def confirm_vacation(auth, gap_id):
    session = http.get_session(auth)
    response = session.post(url=RESOURCES['confirm_api'].format(gap_id=gap_id))
    return response.ok


@helpers.timeit
def get_gaps(auth, logins, **params):
    session = http.get_session(auth)
    request_data = {
        'person_logins': logins,
    }
    for key, val in params.items():
        if key.startswith('date_'):
            params[key] = val.isoformat()
    request_data.update(params)
    raw_data = session.post(
        url=RESOURCES['gaps'],
        json=request_data,
    ).json()
    grouped_gaps = raw_data['persons']
    for person, person_gaps in grouped_gaps.items():
        grouped_gaps[person] = map(Gap, person_gaps)
    return grouped_gaps


# обертки/утилиты для sources нужно поперекладывать
class Gap(object):
    def __init__(self, raw):
        self.raw = raw

    @property
    def date_from(self):
        return datetimes.datetime_to_date(self.raw['date_from'])

    @property
    def date_to(self):
        return datetimes.datetime_to_date(self.raw['date_to'])

    @property
    def first_day(self):
        return self.date_from

    @property
    def is_full_day(self):
        # считаем, что у гэпа по дефолту full_day=True
        return self.raw.get('full_day', True)

    @property
    def last_day(self):
        last_day = self.date_to
        # потому что date_to в гэпе это 00:00
        # следующего за последним дня
        if self.is_full_day:
            last_day = datetimes.shifted_date(last_day, days=-1)
        return last_day

    @property
    def duration(self):
        return (self.date_to - self.date_from).days

    def intersection_with(self, range_start=None, range_end=None):
        if range_start is not None:
            intersection_start = max(self.first_day, range_start)
        else:
            intersection_start = self.first_day
        if range_end is not None:
            intersection_end = min(self.last_day, range_end)
        else:
            intersection_end = self.last_day
        if intersection_end < intersection_start:
            return 0
        return (intersection_end - intersection_start).days + 1

    def intersects_with(self, range_start, range_end):
        return self.intersection_with(range_start, range_end) > 0

    def __getitem__(self, item):
        return self.raw[item]

    def __repr__(self):
        person = self.raw.get('person')
        return '<Gap % s for %s>' % (
            self.raw.get('id'),
            person and person['login'],
        )
