# coding: utf-8

from __future__ import unicode_literals

import logging

from ids.connector import HttpConnector
from ids.connector import plugins_lib
from ids.lib import pagination
from ids.lib import static_api

from cab.utils import env
from cab.utils import auth
from cab.utils import helpers
from cab.utils import dicts


HOST = env.choose({
    'production': 'staff.yandex-team.ru',
    'testing': 'staff.test.yandex-team.ru',
    # 'development': 'port-9100.staff01.dev.yandex-team.ru',
})

PREPROFILE_URL = ''.join((
    'https://',
    HOST,
    '/preprofile/{preprofile_id}',
))


log = logging.getLogger(__name__)


class StaffConnector(HttpConnector):

    available_auth_types = (
        'oauth',
        'session_id',
    )

    plugins = (
        auth.MultiAuthPlugin,
        plugins_lib.JsonResponse,
    )

    url_patterns = {
        'saved_filter_list': '/filters-api/saved-filters/?is_for_cab=1',
        'saved_filter_persons': '/filters-api/saved-filters/{filter_id}/persons/',
        'dynamic_filter': '/filters-api/filter/persons/',
        'last_activity': '/whistlah/where/',
        'personal_data': '/profile-api/bulk/',
        'person_lenta': '/lenta-api/lenta/{login}/',
        'preprofiles_to_confirm': '/preprofile-api/export/cab/',
    }

    post_resources = (
        'dynamic_filter',
    )

    def execute_request(self, method='get', resource=None, *args, **kwargs):
        if resource in self.post_resources:
            method = 'post'
        return super(StaffConnector, self).execute_request(
            method, resource, *args, **kwargs)


class DummyEmptyResultSet(pagination.ResultSet):

    def __init__(self):
        super(DummyEmptyResultSet, self).__init__(fetcher=None)

    def get_pages(self):
        return []

    @property
    def first_page(self):
        return None

    @property
    def total(self):
        return 0

    @property
    def pages(self):
        return 0


def _get_connector(auth):
    return StaffConnector(
        host=HOST,
        protocol='https',
        auth=auth,
        user_agent='staff-cab'
    )


@helpers.timeit
def get_preprofiles_to_confirm(auth):
    return _get_connector(auth).get(
        resource='preprofiles_to_confirm'
    )


@helpers.timeit
def get_filters(auth):
    connector = _get_connector(auth)
    return connector.get(resource='saved_filter_list')


@helpers.timeit
def get_saved_filter_persons(auth, staff_saved_filter_id, paginator=None):
    connector = _get_connector(auth)
    query_params = {}
    if paginator:
        query_params['_page'] = paginator.page
        query_params['_limit'] = paginator.limit
    result_set = pagination.ResultSet(static_api.StaticApiFetcher(
        connector=connector,
        resource='saved_filter_persons',
        url_vars={'filter_id': staff_saved_filter_id},
        params=query_params,
    ))
    return result_set


@helpers.timeit
def get_direct_subordinates(auth, staff_id, paginator=None):
    query = {
        "filters": [
            {
                "operator": "and",
                "field": "SubordinatesFilter",
                "condition": "Direct",
                "person": staff_id,
            }
        ]
    }
    return get_dynamic_filter_persons(auth, query, paginator)


@helpers.timeit
def get_all_subordinates(auth, staff_id, paginator=None):
    query = {
        "filters": [
            {
                "operator": "and",
                "field": "SubordinatesFilter",
                "condition": "All",
                "person": staff_id,
            }
        ]
    }
    return get_dynamic_filter_persons(auth, query, paginator)


@helpers.timeit
def get_department_persons(auth, department_id, paginator=None):
    query = {
        "filters": [
            {
                "operator": "and",
                "field": "DepartmentFilter",
                "condition": "InHierarchy",
                "department": department_id,
            }
        ]
    }
    return get_dynamic_filter_persons(auth, query, paginator)


@helpers.timeit
def get_service_group_persons(auth, service_group_id, paginator=None):
    query = {
        "filters": [
            {
                "operator": "and",
                "field": "ServiceFilter",
                "condition": "Equal",
                "group": service_group_id,
            }
        ]
    }
    return get_dynamic_filter_persons(auth, query, paginator)


@helpers.timeit
def get_dynamic_filter_persons(auth, query, paginator=None):
    connector = _get_connector(auth)
    query_params = {}
    if paginator:
        query_params['_page'] = paginator.page
        query_params['_limit'] = paginator.limit
    result_set = pagination.ResultSet(static_api.StaticApiFetcher(
        connector=connector,
        resource='dynamic_filter',
        json=query,
        params=query_params,
    ))
    return result_set


@helpers.timeit
def get_last_activity(auth, logins):
    connector = _get_connector(auth)
    query_params = {'logins': ','.join(logins)}
    last_activity_list = connector.get(
        resource='last_activity',
        params=query_params,
    )
    last_activity_map = {
        la['person']['login']: la
        for la in last_activity_list
    }
    return last_activity_map


@helpers.timeit
def get_personal_fields(auth, logins, paginator=None):
    connector = _get_connector(auth)
    query_params = {
        'logins': ','.join(logins),
        'fields': ['all'],
    }
    if paginator:
        query_params['_page'] = paginator.page
        query_params['_limit'] = paginator.limit
    result_set = pagination.ResultSet(static_api.StaticApiFetcher(
        connector=connector,
        resource='personal_data',
        params=query_params,
    ))
    return dicts.dictify(result_set, key='login')


@helpers.timeit
def get_person_lenta(auth, login):
    connector = _get_connector(auth)
    url_vars = {'login': login}
    response = connector.get(
        resource='person_lenta',
        url_vars=url_vars,
    )
    return response
