# coding: utf-8

from __future__ import unicode_literals

import functools
import logging

from django.conf import settings
from ids.exceptions import BackendError
from ids.registry import registry
from ids.services.staff import connector
from ids.lib import static_api
from ids.lib import pagination
from ids.connector import plugins_lib

from cab.utils import auth as auth_utils, env, std
from cab.utils import http
from cab.utils import helpers
from cab.utils import dicts

URL = env.choose({
    'production': 'https://staff-api.yandex-team.ru/v3',
    'testing': 'https://staff-api.test.yandex-team.ru/v3',
})

RESOURCES = {
    'persons': URL + '/persons',
    'groups': URL + '/groups',
}

log = logging.getLogger(__name__)


class StaffApiConnector(connector.StaffConnector):

    default_connect_timeout = 10

    available_auth_types = (
        'oauth',
        'session_id',
    )

    plugins = (
        auth_utils.MultiAuthPlugin,
        plugins_lib.JsonResponse,
    )


def get_result_set(auth, resource, **request_params):
    connector = StaffApiConnector(
        auth=auth,
        user_agent='cab',
    )
    fetcher = static_api.StaticApiFetcher(
        connector=connector,
        resource=resource,
        **request_params
    )
    return pagination.ResultSet(fetcher=fetcher)


PERSON_IDENTIFIEDS = (
    'uid',
    'login',
    'staff_id',
)


class PersonIdentifier(object):

    def __init__(self, **identifier):
        if len(identifier) != 1:
            raise RuntimeError('Bad person identifier: %s', identifier)
        self.key, self.value = identifier.items()[0]
        if self.key not in PERSON_IDENTIFIEDS:
            raise RuntimeError('Bad person identifier key: %s' % self.key)

    def as_dict(self):
        return {self.key: self.value}

    def as_tuple(self):
        return self.key, self.value


def fetch_person_id(func):
    @functools.wraps(func)
    def deco(*args, **kwargs):
        person_id = None
        for key, value in kwargs.items():
            if key in PERSON_IDENTIFIEDS:
                if person_id is not None:
                    raise RuntimeError('Too many person identifiers %s', kwargs)
                person_id = PersonIdentifier(**{key: kwargs.pop(key)})
        if person_id is None:
            raise RuntimeError('No person_identifier given %s', kwargs)
        kwargs['person_id'] = person_id
        return func(*args, **kwargs)
    return deco


def memoize_person_data(func):
    func._memoized = {}

    def deco(*args, **kwargs):
        person_id = kwargs['person_id']
        if person_id.as_tuple() not in func._memoized:
            person_data = func(*args, **kwargs)
            if person_data:
                func._memoized[('staff_id', person_data['id'])] = person_data
                func._memoized[('login', person_data['login'])] = person_data
                func._memoized[('uid', person_data['uid'])] = person_data
            else:
                return person_data
        return func._memoized[person_id.as_tuple()]

    return deco


@fetch_person_id
@memoize_person_data
@helpers.timeit
def get_person_data(person_id):
    repo = registry.get_repository(
        service='staff',
        resource_type='person',
        user_agent='cab',
        oauth_token=settings.CAB_OAUTH_TOKEN,
        timeout=settings.STAFF_API_TIMEOUT,
    )
    query_params = {'_one': 1}
    query_params.update(**person_id.as_dict())

    try:
        return repo.get_one(query_params)
    except BackendError:
        log.error('Cannot get person data from Staff-API for login `%s`', query_params['login'])


@helpers.timeit
def get_persons_data(auth, logins, fields=None):
    return get_persons_data_by_filter(
        auth=auth,
        filters={'login': ','.join(logins)},
        fields=fields,
        group_key='login',
    )


@helpers.timeit
def get_subordinate_departments(auth, uid=None, login=None, depth=0, fields=None):
    if login is None and uid is None:
        raise Exception('login or uid should be given')

    if uid:
        id_name = 'uid'
        id = uid
    else:
        id_name = 'login'
        id = login

    if depth == 0:
        key = 'department.heads.person.' + id_name
    elif depth == 1:
        key = 'parent.department.heads.person.' + id_name
    else:
        raise Exception("Too deep")

    fields = fields or 'department.name,department.id,department.url'
    query_params = {
        key: id,
        '_fields': fields,
    }
    return get_groups_data(auth, query_params)


@helpers.timeit
def get_service_group(auth, service_id=None, slug=None, fields=None):
    if service_id is None and slug is None:
        raise Exception('service_id or slug should be given')

    if service_id:
        key = 'service.id'
        id = service_id
    else:
        key = 'url'
        id = 'svc_' + slug

    groups = get_groups_data(
        auth=auth,
        query_params={
            key: id,
            '_fields': fields or 'id,slug,name',
        },
    )
    return groups and groups[0] or None


@helpers.timeit
def get_subordinate_persons(auth, id_, fields=None):
    filters = {
        '_query': " ".join([
            "(",
            "department_group.department.heads.person.id == %d" % id_,
            "or",
            "department_group.ancestors.department.heads.person.id == %d" % id_,
            ")",
            "and id != %d" % id_,
        ]),
        'official.is_robot': False,
        'official.is_dismissed': False,
    }
    return get_persons_data_by_filter(
        auth=auth,
        filters=filters,
        fields=fields,
    )


@helpers.timeit
def get_persons_data_by_filter(auth, filters, fields=None, group_key=None):
    query_params = filters.copy()
    fields = fields or []
    if group_key and group_key not in fields:
        fields.append(group_key)
    query_params['_fields'] = ','.join(fields)
    query_params['_limit'] = 500

    data = get_result_set(
        auth=auth,
        resource='person',
        params=query_params,
    )

    if group_key:
        return dicts.dictify(data, key=group_key)
    else:
        return data


@helpers.timeit
def get_groups_data(auth, query_params):
    session = http.get_session(auth)
    return session.get(
        url=RESOURCES['groups'],
        params=query_params,
    ).json()['result']
