# coding: utf-8

from __future__ import unicode_literals

import logging
import operator
import decimal

import requests

from cab.utils import cache
from cab.utils import datetimes


log = logging.getLogger(__name__)

MORDA_YNDX_NASDAQ_ID = 42
MORDA_USD_CENTRAL_BANK_ID = 10000


def get_all_morda_stocks():
    response = requests.get('http://stocks.yandex.net/morda-v2.json')
    if response.ok:
        return response.json()['stocks']
    log.error(
        'Morda stocks fetch failed with http status %s',
        response.status_code
    )
    return []


@cache.cached_safe(
    timeout=cache.TEN_MINUTES_IN_SECONDS,
    backend='locmem',
    key='get_interesting_stocks',
)
@cache.cached_safe(
    timeout=cache.ONE_DAY_IN_SECONDS,
    backend='db',
    key='get_interesting_stocks',
)
def get_interesting_stocks():
    morda_json = get_all_morda_stocks()
    return {
        'usd_in_rub': _select_value(morda_json, id=MORDA_USD_CENTRAL_BANK_ID),
        'yndx_in_usd': _select_value(morda_json, id=MORDA_YNDX_NASDAQ_ID),
    }


def _select_value(all_data, id, key='bv'):
    today_morda_fmt = datetimes.now().isoformat(sep=str(' '))
    all_currencies_for_id = sorted(
        [
            datum for datum in all_data
            if datum['id'] == id and datum['dt'] <= today_morda_fmt
        ],
        key=operator.itemgetter('dt')
    )
    if not all_currencies_for_id:
        return
    last_data_item = all_currencies_for_id[-1]
    return last_data_item.get(key)


def get_usd_to_ruble_rate():
    default = '60'
    val = get_interesting_stocks() and get_interesting_stocks()['usd_in_rub']
    return decimal.Decimal(val or default)


def get_yndx_usd_stock_price():
    default = '20'
    val = get_interesting_stocks() and get_interesting_stocks()['yndx_in_usd']
    return decimal.Decimal(val or default)


def get_yndx_rub_stock_price():
    return get_yndx_usd_stock_price() * get_usd_to_ruble_rate()
