# coding: utf-8

from __future__ import unicode_literals

import functools
import inspect
import logging
import re
from datetime import datetime
from itertools import chain

from django.conf import settings

log = logging.getLogger(__name__)


def fail_silently(cls=Exception, log_exception=True):
    def deco(func):
        @functools.wraps(func)
        def wrapper(*args, **kwargs):
            try:
                return func(*args, **kwargs)
            except cls:
                if log_exception:
                    log.exception('%s failed silently', func.__name__)
        return wrapper
    return deco


def timeit(func):
    fu_arg_names = tuple(inspect.getargspec(func).args)

    @functools.wraps(func)
    def deco(*args, **kwargs):
        before = datetime.now()
        msg = '%d.%06d TIMEIT for %s %s %s(%s)'
        try:
            return func(*args, **kwargs)
        except Exception as exc:
            msg += ' with exception %s' % exc.__class__.__name__
            raise
        finally:
            mark_type = 'unknown'
            mark_id = func.__module__
            if mark_id.startswith('cab.sources.'):
                mark_type = 'source'
                mark_id = mark_id[len('cab.sources.'):]
            elif mark_id.startswith('cab.widgets.'):
                mark_type = 'endpoint'
                mark_id = func.im_class.__name__
            elif mark_id.startswith('ids'):
                mark_type = 'ids'

            passed = datetime.now() - before
            kwargs_to_log = get_kwargs_to_log(fu_arg_names, args, kwargs)
            args_str = ', '.join(
                '{}:{}'.format(k, v)
                for k, v in kwargs_to_log
            )
            log.info(
                msg,
                passed.seconds,
                passed.microseconds,
                mark_type,
                mark_id,
                func.__name__,
                args_str,
            )
    return deco


def get_kwargs_to_log(arg_names, fu_args, fu_kwargs):
    kwargs_to_log = chain(zip(arg_names, fu_args), fu_kwargs.items())
    allowed_to_log = get_allowed_to_log(arg_names)
    return (
        (k, _prepare_for_log(v)) for k, v in kwargs_to_log
        if k in allowed_to_log
    )


_cached_allowed_to_log = {}


def get_allowed_to_log(arg_names):
    arg_names = tuple(arg_names)
    global _cached_allowed_to_log
    cached = _cached_allowed_to_log.get(arg_names)
    if cached is not None:
        return cached
    forbidden_matchers = [re.compile(it) for it in settings.FORBIDDEN_TO_LOG_REGEX]
    allowed_matchers = [re.compile(it) for it in settings.FORCE_ALLOWED_TO_LOG]
    res = {
        arg_name for arg_name in arg_names
        if (
            any(r.match(arg_name) is not None for r in allowed_matchers) or
            all(r.match(arg_name) is None for r in forbidden_matchers)
        )
    }
    _cached_allowed_to_log[arg_names] = res
    return res


def _prepare_for_log(obj):
    if isinstance(obj, basestring):
        return "'%s'" % obj
    else:
        return str(obj)
