# coding: utf-8
from __future__ import unicode_literals

import logging
import functools
import multiprocessing

import attr
from ids import exceptions as ids_exceptions

log = logging.getLogger(__name__)


@attr.s
class Content(object):
    value = attr.ib(default=None)

    is_content = True
    is_error = False


@attr.s
class Error(object):
    value = attr.ib(default=None)

    is_content = False
    is_error = True


def fetcher(func):
    @functools.wraps(func)
    def wrapper(*args, **kwargs):
        try:
            return Content(value=func(*args, **kwargs))
        except ids_exceptions.BackendError as exc:
            response_text = exc.response is not None and exc.response.content or ''
            log.exception(
                'Fetch %s failed with BackendError: [%s] %s',
                func.__name__,
                exc.status_code or '???',
                response_text,
            )
            return Error(value=str(exc))
        except Exception as exc:
            log.exception('Fetch %s failed', func.__name__)
            return Error(value=str(exc))
    return wrapper


def collect_data(env_prefix, func, args_dict):
    import os
    mode = os.environ.get(env_prefix.upper() + '_FETCH_MODE')
    mode = mode or os.environ.get('FETCH_MODE') or 'serial'
    log.debug('FETCH %s in %s mode', env_prefix, mode)
    return fetch_multi(func=func, args_dict=args_dict, mode=mode)


def fetch_multi(func, args_dict, mode='parallel'):
    if mode == 'parallel':
        return fetch_parallel(func, args_dict)
    else:
        return fetch_serial(func, args_dict)


def fetch_parallel(func, args_dict):
    pool = multiprocessing.Pool(processes=len(args_dict))

    results = pool.map(func, [
        (resource_id,) + tuple(args)
        for resource_id, args in args_dict.items()
    ])
    pool.close()
    pool.join()

    return {key: result for key, result in zip(args_dict, results)}


def fetch_serial(func, args_dict):
    return {
        resource_id: func((resource_id,) + tuple(args))
        for resource_id, args in args_dict.items()
    }
