# coding: utf-8

from __future__ import unicode_literals

import logging
import requests

from django.conf import settings


log = logging.getLogger(__name__)

YANDEX_NV_COMPANY_ID = 202  # CAB-600


def replace_class_codes(grants):
    class_codes_to_abbr = {
        'N1R': 'RSU',
        'N1U': 'RSU',
        'N2R': 'OPT',
        'N2U': 'OPT',
        'RS1': 'RSU',
        'SSR': 'SAR',
    }

    for grant in grants:
        class_code = grant['classCode']
        grant['classCode'] = class_codes_to_abbr.get(class_code, class_code)


def get_stock_options(login, need_all_companies):
    """
    Не валидируем логин, так как в oebs могут быть логины которых у нас нет
    и нам надо это поддержать
    """
    request = {
        'login': [login],
        'showGrants': {
            'declared': 'Y',
            'grantStatus': [
                'AGREED',
                'PRE_AGREED',
                'NEW',
                'UPLOADED',
            ],
        }

    }
    if not need_all_companies:
        request['showGrants']['company'] = [YANDEX_NV_COMPANY_ID]

    try:
        response = requests.post(
            url=settings.OEBS_REST_API + '/options',
            headers={'Authorization': 'OAuth %s' % settings.CAB_OAUTH_TOKEN},
            json=request,
            timeout=12,
        )
    except (requests.Timeout, requests.ConnectTimeout):
        err_msg = 'OEBS is unavailable'
        log.exception(err_msg + ' for %s', login)
        return dict(error=err_msg)

    try:
        result = response.json()
    except ValueError:
        err_msg = 'OEBS returned invalid json'
        log.exception(err_msg + ' for %s', login)
        return dict(error=err_msg)

    if not result.get('holders', []):
        return {'grants': []}

    grants = result['holders'][0].get('grants', [])

    replace_class_codes(grants)

    return {'grants': grants}


def push_plan(login, plan_data):
    url = '{api}pushPlanInsider'.format(api=settings.OEBS_REST_API)
    plan_data['login'] = login
    try:
        return requests.post(
            url=url,
            headers={'Authorization': 'OAuth %s' % settings.CAB_OAUTH_TOKEN},
            json=plan_data,
        ).json()
    except Exception:
        err_msg = 'Error trying to push plan data for {login}'.format(login=login)
        log.exception(err_msg)
        raise


def get_plan(logins):
    if not logins:
        return {}

    url = '{api}getPlanInsider'.format(api=settings.OEBS_REST_API)
    request_data = {'login': logins}
    try:
        return requests.post(
            url=url,
            headers={'Authorization': 'OAuth %s' % settings.CAB_OAUTH_TOKEN},
            json=request_data,
        ).json()
    except Exception:
        err_msg = 'Error trying to get plan data for {logins}'.format(logins=logins)
        log.exception(err_msg)
        raise
