# coding: utf-8

from __future__ import unicode_literals

import logging
import itertools

from django import forms

from cab.core import widgets
from cab.sources import (
    gap,
    staff,
    startrek,
)
from cab.utils import dehydration, datetimes, dicts, env, multifetch
from cab.utils import validators

log = logging.getLogger(__name__)


QUEUES = {
    'salary': {
        'query_params': {
            'queue': env.choose(
                {
                    'production': 'SALARY',
                    'testing': 'TSALARY',
                }
            ),
            'resolution': 'empty()',
        },
        'fields': [
            'key',
            'summary',
            'createdBy',
        ],
        'display_person': 'createdBy',
    },
    'job': {
        'query_params': {
            'queue': env.choose(
                {
                    'production': 'JOB',
                    'testing': 'TJOB',
                }
            ),
            'resolution': 'empty()',
        },
        'fields': [
            'key',
            'summary',
            'createdBy',
        ],
        'display_person': 'createdBy',
    },
    'conference_trip': {
        'query_params': {
            'queue': 'INTERCONF',
            'status': 'open',
        },
        'fields': [
            'key',
            'summary',
            'cities',
            'employee',
        ],
        # 'fields': 'key,summary,cities,startDate,endDate,employee',
        'display_person': 'employee',
    },
    'conference': {
        'query_params': {
            # та же очередь, что и conference_trip
            # упоминаем здесь, но не добавляем лишнего в запрос
        },
        'fields': [
            'key',
            'summary',
            'cities',
            'employee',
        ],
        # 'fields': 'key,summary,cities,startDate,endDate,employee',
        'display_person': 'employee',
    },
    'trip': {
        'query_params': {
            'queue': 'TRAVEL',
            'status': 'open',
        },
        'fields': [
            'key',
            'summary',
            'itinerary',
            'startDate',
            'endDate',
            'employee',
        ],
        'display_person': 'employee',
    },
    'approval': {
        'query_params': {},
        'fields': ['employee'],
        'display_person': 'employee'
    }
}


def get_issues_type(issue):
    queue, id_ = issue['key'].split('-')
    if queue == 'INTERCONF':
        # https://st.yandex-team.ru/STAFF-6935
        if issue['cities']:
            return 'conference_trip'
        else:
            return 'conference'
    else:
        return {
            params['query_params']['queue']: action_type
            for action_type, params
            in QUEUES.items()
            if dicts.get_nested(params, 'query_params.queue') is not None
        }.get(queue, 'approval')


class AvailableActionsList(widgets.BaseEndpointHandler):

    def handle(self, data):
        args_dict = {
            service_id: (self.auth,)
            for service_id in (
                'startrek',
                'preprofile',
                'gap',
            )
        }
        raw_data = multifetch.collect_data(
            env_prefix='actions',
            func=fetch,
            args_dict=args_dict,
        )

        content = []
        for service_result in raw_data.values():
            if service_result.is_content:
                content.extend(service_result.value)
        return {'content': content}


@multifetch.fetcher
def fetch(args):
    service_id, func_args = args[0], args[1:]
    return globals()['get_' + service_id + '_data'](*func_args)


def get_startrek_data(auth):
    results = []

    query = ' or '.join([
        '(%s)' % ' and '.join([
            '%s: %s' % (key, val)
            for key, val in action_type_params['query_params'].items()
            if action_type_params.get('query_params')
        ])
        for action_type_params in QUEUES.values()
    ])
    query = '(%s) and assignee: me()' % query
    query = '(%s) or ("Approvement Status": Запущено AND "Current Approvers": me())' % query

    fields = set(itertools.chain.from_iterable([
        params['fields'] for params in QUEUES.values()
    ]))

    st_data = startrek.get_issues_by_query(
        auth=auth,
        query=query,
        fields=','.join(fields),
    )
    for issue in st_data:
        key = issue['key']
        action_type = get_issues_type(issue)
        st_params = QUEUES[action_type]

        action_data = {
            'type': action_type,
            'data': {
                'issue': key,
                'url': startrek.RESOURCES['issue_web'].format(issue_key=key),
                'person': None,
            }
        }

        if 'startDate' in st_params['fields']:
            date_from = issue.get('startDate')
            date_to = issue.get('endDate')

            if date_from or date_to:
                action_data['data']['dates'] = {
                    'from': date_from and datetimes.convert_date_format(date_from),
                    'to': date_to and datetimes.convert_date_format(date_to),
                }

        display_person_field = st_params['display_person']
        display_person = issue[display_person_field]
        employee_data = {}
        if display_person:
            first_name, last_name = startrek.parse_name(
                display_person['display'])
            employee_data = {
                'first_name': first_name,
                'last_name': last_name,
                'login': display_person['id'],
            }
        person = dehydration.robot_startrek(**employee_data)
        action_data['data']['person'] = person

        if 'cities' in st_params['fields']:
            cities = issue.get('cities')
            if cities:
                action_data['data']['city'] = cities

        description = issue.get('itinerary') or issue['summary']
        action_data['data']['description'] = description

        results.append(action_data)
    return results


def get_preprofile_data(auth):
    preprofiles = staff.get_preprofiles_to_confirm(auth=auth).get('preprofiles', [])
    robot_staff = dehydration.robot_staff()
    return [
        {
            'type': 'offer',
            'id': pr['id'],
            'data': {
                'person': robot_staff,
                'url': staff.PREPROFILE_URL.format(preprofile_id=pr['id']),
                'new_employee': {
                    'first_name': pr['first_name'],
                    'last_name': pr['last_name'],
                }
            }
        }
        for pr in preprofiles
    ]


def get_gap_data(auth):
    vacations = gap.get_vacations(auth=auth)
    results = []

    today = datetimes.today()
    for vacation in vacations:
        person = dehydration.user(
            first_name=vacation['person']['first_name'],
            last_name=vacation['person']['last_name'],
            login=vacation['person']['login'],
        )
        if vacation.last_day < today:
            continue

        results.append({
            'type': 'vacation',
            'data': {
                'person': person,
                'dates': {
                    'from': vacation.first_day.isoformat(),
                    'to': vacation.last_day.isoformat(),
                },
                'id': vacation['id'],
                'url': gap.RESOURCES['confirm_web'].format(gap_id=vacation['id']),
                'comment': vacation['comment'],
            }
        })
    return results


class ProcessAction(widgets.BaseEndpointHandler):

    class Validator(validators.WidgetValidator):
        id = forms.IntegerField()
        type = forms.ChoiceField(choices=(
            ('vacation', 'vacation'),
        ))
        action = forms.ChoiceField(choices=(
            ('approve', 'approve'),
        ))

    def handle(self, data):
        id = data.get('id')
        return {
            'status': gap.confirm_vacation(auth=self.auth, gap_id=id)
        }


class ActionsWidget(widgets.BaseWidget):
    id = 'actions'

    endpoints = {
        'root': {
            'regex': '/$',
            'handler': {
                'GET': AvailableActionsList,
                'POST': ProcessAction,
            }
        },
    }

