# coding: utf-8

from __future__ import unicode_literals

import operator
import logging

from ids import exceptions

from cab.core import widgets
from cab.sources import staff
from cab.sources import staff_api
from cab.sources import review
from cab.sources import startrek_goals
from cab.utils import datetimes
from cab.utils import dehydration
from cab.utils import dicts
from cab.utils import multifetch
from cab.utils import l10n
from cab.utils import validators

log = logging.getLogger(__name__)


class PersonHistory(widgets.BaseEndpointHandler):
    class Validator(validators.WidgetValidator):
        login = validators.LoginField(required=False)

    def handle(self, data):
        login = data['login'] or self.auth.login
        staff_id = staff_api.get_person_data(login=login)['id']

        services = {
            'lenta': (self.auth, login),
            'review': (self.auth, login),
            'goals': (self.auth, staff_id),
            'finance': (self.auth, login),
        }

        data = multifetch.collect_data(
            env_prefix='history',
            func=fetch,
            args_dict=services,
        )

        events = []
        errors = []
        for service_id, service_result in data.items():
            if service_result.is_content:
                events.extend(service_result.value)
            else:
                errors.append(service_id)
        events = sorted(events, key=operator.itemgetter('date'))
        events = skip_trash_events(events)
        events = squash_related_events(events)
        events = list(reversed(events))
        return [
            {
                'type': 'error',
                'service': service,
            }
            for service in errors
        ] + events


@multifetch.fetcher
def fetch(args):
    service_id, func_args = args[0], args[1:]
    return globals()['get_' + service_id + '_data'](*func_args)


def get_lenta_data(auth, login):
    lenta = staff.get_person_lenta(auth, login)
    lenta = map(l10n.localized, lenta)
    for event in lenta:
        event['type'] = event.pop('action').lower()
    return lenta


def get_review_data(auth, login):
    try:
        history = review.get_person_reviews_announced(auth, [login])
    except exceptions.AuthError:
        return []

    history = history.get(login, [])

    results = []
    for event in history:
        if not event['mark'] or event['mark'] == review.DISABLED:
            continue
        history_event = {
            'type': 'review',
            'mark': event['mark'],
            'goldstar': review.get_goldstar_value(event['goldstar']),
            'date': event['review']['finish_date'],
            'review_type': event['review']['type'],
            'review_id': event['review']['id'],
            'person_review_id': event['id'],
        }
        results.append(history_event)
    return results


def get_finance_data(auth, login):
    fields = (
        'salary_history',
        'grade_history',
        'bonus_history',
        'current_loans',
        'options_history',
    )
    history = review.get_oebs_data(
        auth=auth,
        fields=fields,
        logins=[login]
    )
    history = history.get(login)

    if history is None:
        return []

    result = []
    for field in fields:
        value = history.get(field)
        if value is None:
            continue
        if value == review.NO_ACCESS:
            continue
        result.extend(globals()['parse_%s' % field](value))
    return result


def parse_salary_history(history):
    history = sorted(history, key=operator.itemgetter('dateFrom'))

    result = []
    previous_salary = None
    for event in history:
        current_salary = dehydration.salary(event)
        result.append({
            'type': 'salary',
            'date': event['dateFrom'],
            'before': previous_salary,
            'after': current_salary
        })
        previous_salary = current_salary

    return result


def parse_grade_history(history):
    history = sorted(history, key=operator.itemgetter('dateFrom'))

    result = []
    previous_grade = None
    for event in history:
        current_grade = dehydration.grade(event)
        result.append({
            'type': 'grade',
            'date': event['dateFrom'],
            'before': previous_grade,
            'after': current_grade,
        })
        previous_grade = current_grade

    return result


def parse_bonus_history(history):
    return [
        {
            'type': 'bonus',
            'payment': {
                'value': event['summ'],
                'currency': event['currency'],
            },
            'date': event['effectiveDate'],
        }
        for event in history
    ]


def parse_current_loans(history):
    return [
        {
            'type': 'loan',
            'loan': {
                'value': event['loanSum'],
                'currency': event['currency'],
                'loan_type': int(event['loanType']),
            },
            'date': event['contractDate'],
        }
        for event in history
    ]


def parse_options_history(history):
    return [
        {
            'type': 'option',
            'option': {
                'value': event['grantAmount'],
                'grant_type': event['classCode'],
            },
            'date': event['grantDate'],
        }
        for event in history
    ]


def get_goals_data(auth, staff_id):
    login = staff_api.get_person_data(staff_id=staff_id)['login']
    resolved_goals = startrek_goals.get_resolved_goals(auth=auth, login=login)

    results = []
    for goal in resolved_goals:
        formatted_goal = startrek_goals.format_goal_for_response(goal)
        resolve_datetime = goal['statusStartTime']
        resolve_date = resolve_datetime and resolve_datetime[:len('YYYY-MM-DD')]
        formatted_goal.update(
            type='goals',
            date=resolve_date,
        )
        results.append(formatted_goal)
    return results


def skip_trash_events(events):
    return [
        event for event in events
        if not is_trash_event(event)
    ]


def is_trash_event(event):
    if 'before' not in event or 'after' not in event:
        return False

    if event['type'] == 'grade':
        # @aleksundra: давай из истории изменений грейдов исключать события,
        # когда менялся не сам грейд, а профессия.
        level_before = event['before'] and event['before']['level']
        level_after = event['after'] and event['after']['level']
        return level_before == level_after
    else:
        return event['before'] == event['after']


REVIEW_PAYMENTS_GAP_DAYS = 30


def is_relevant_to_review(event, review_event):
    type = event['type']
    if type == 'bonus':
        if 'bonus_payment' in review_event:
            return False

        days = abs(datetimes.days_between(event['date'], review_event['date']))
        if days < REVIEW_PAYMENTS_GAP_DAYS:
            return True

    if type in ('salary', 'grade'):
        if type == 'salary' and 'salary_after_review' in review_event:
            return False
        if type == 'grade' and 'grade_after_review' in review_event:
            return False

        days = abs(datetimes.days_between(event['date'], review_event['date']))
        if days < REVIEW_PAYMENTS_GAP_DAYS:
            return True
    return False


def append_to_review(event, review_event):
    if event['type'] == 'bonus':
        review_event['bonus_payment'] = event['payment']
    if event['type'] == 'salary':
        after = event['after']
        before = event['before']
        if before['value'] and after['value']:
            salary_change_ratio = float(after['value'] - before['value']) / before['value']
            salary_change_percentage = int(salary_change_ratio * 100)
        else:
            salary_change_percentage = 0
        review_event.update(
            salary_before_review=event['before'],
            salary_after_review=event['after'],
            salary_change_percentage=salary_change_percentage,
        )
    if event['type'] == 'grade':
        review_event.update(
            grade_before_review=event['before'],
            grade_after_review=event['after'],
        )


def squash_related_events(events):
    results = []

    review_ranges = dicts.RangeDict.from_gapless_range([
        (event['date'], event['after'])
        for event in events
        if event['type'] == 'grade'
    ])

    last_review = None
    for event in events:
        if event['type'] == 'review':
            last_review = event
        if last_review and is_relevant_to_review(event, last_review):
            append_to_review(event, last_review)
            continue
        results.append(event)

    for event in results:
        if event['type'] == 'review' and 'grade_after_review' not in event:
            grade_for_date = review_ranges.get(event['date'])
            append_to_review(
                event={
                    'type': 'grade',
                    'before': grade_for_date,
                    'after': grade_for_date,
                },
                review_event=event,
            )

    return results


class HistoryWidget(widgets.BaseWidget):
    id = 'history'

    endpoints = {
        'root': {
            'regex': '/$',
            'handler': PersonHistory,
        }
    }
