from django.db.models.query import QuerySet
from django.conf import settings

from rest_framework.pagination import CursorPagination


class CompositorPagination(CursorPagination):
    ordering = settings.CURSOR_ORDERING
    cursor_query_param = 'page'
    page_size_query_param = 'per_page'
    max_page_size = 100

    def paginate_queryset(self, queryset, request, view=None):
        if not isinstance(queryset, QuerySet):
            queryset = Container(queryset)
        return super().paginate_queryset(queryset, request, view)


class Container:
    """
    Пагинация курсором рассчитана на работу с
    queryset, реализуем в этом классе нужные
    методы для работы пагинации курсором с iterable
    """
    def __init__(self, values):
        self.values = values

    def order_by(self, *args, **kwargs):
        return Container(
            sorted(
                self.values, key=lambda obj: obj[settings.CURSOR_ORDERING]
            )
        )

    def filter(self, **kwargs):
        ordering_param, ordering_value = kwargs.popitem()
        direction = ordering_param.split('__')[1]
        order_func = lambda obj: obj[settings.CURSOR_ORDERING] > ordering_value
        if direction == 'lt':
            order_func = lambda obj: obj[settings.CURSOR_ORDERING] < ordering_value

        return list(filter(order_func, self.values))

    def __iter__(self):
        return iter(self.values)

    def __getitem__(self, index):
        return self.values[index]
