import logging
import nirvana_api

from django.conf import settings

from rest_framework import generics
from rest_framework.response import Response
from rest_framework.exceptions import APIException

from django_tools_log_context import request_context, request_profiler

from .. import errors
from .. import pagination

logger = logging.getLogger(__file__)

NIRVANA_NOT_FOUND_CODE = 90


class APIView(generics.GenericAPIView):
    """
    Базовый класс для всех API View
    """

    def dispatch(self, request, *args, **kwargs):
        with request_context(request, endpoint=self), request_profiler(request, threshold=settings.THRESHOLD):
            response = super().dispatch(request, *args, **kwargs)

        return response

    def handle_exception(self, exc):
        """
        Обработать исключения из хэндлера запроса
        @param exc: объект исключения
        """

        if isinstance(exc, errors.RestApiError):
            data = errors.rest_api_error_to_dict(exc)
            response = Response(data, status=exc.status_code, exception=True)

        elif isinstance(exc, APIException):
            if isinstance(exc.detail, dict):
                exc_data = exc.detail
            else:
                if isinstance(exc.detail, (list, tuple)):
                    detail = exc.detail
                else:
                    detail = [exc.detail]
                exc_data = {'detail': detail}
            data = {
                'error_code': errors.UnhandledException.error_code,
                'level': 'ERROR',
                'message': [errors.get_error_message(exc)],
                'debug_message': exc.default_detail,
                'errors': exc_data,
            }
            response = Response(data, status=exc.status_code, exception=True)
        else:
            logger.exception('Error happened while handling request "%s"', repr(exc))
            # 500я ошибка
            data = {
                'error_code': errors.UnhandledException.error_code,
                'level': 'ERROR',
                'debug_message': '{0}: {1}'.format(
                    errors.UnhandledException.debug_message, repr(','.join(map(str, exc.args)))),
                'message': errors.UnhandledException().non_field_messages,
            }
            response = Response(data, status=500, exception=True)
        logger.info('Responded with "%s"', response.status_code)
        return response


class NirvanaErrorHandlerMixin:
    def handle_exception(self, exc):
        if isinstance(exc, nirvana_api.json_rpc.RPCException):
            logger.error(f'Nirvana api responded with code: {exc.args[0]}, error: {exc.args[1]}')
            if exc.args[0] == NIRVANA_NOT_FOUND_CODE:
                exc = errors.NirvanaNotFoundError()
            else:
                exc = errors.NirvanaApiError(exc.args[1])

        return super().handle_exception(exc)


class ListAPIView(NirvanaErrorHandlerMixin, APIView, generics.ListAPIView):
    pagination_class = pagination.CompositorPagination


class RetrieveAPIView(NirvanaErrorHandlerMixin, APIView, generics.RetrieveAPIView):
    pk_field = 'pk'

    def get(self, request, *args, **kwargs):
        self.pk = kwargs[self.pk_field]
        return super().get(request, *args, **kwargs)
