import json

from rest_framework.response import Response

from . import base
from .base import NirvanaErrorHandlerMixin, APIView
from ..errors import IllegalWorkflowInstanceStatus
from ..serializers.instance import (
    InstanceDetailSerializer
)
from ...clients.mds import MdsS3ApiClient
from ...clients.nirvana import nirvana_client


class InstanceDetailApiView(base.RetrieveAPIView):
    """
    Возвращает информацию о выполнении инстанса воркфлоу
    Если инстанс выполнился и успешно - так же вернет полученные
    результаты
    """
    serializer_class = InstanceDetailSerializer

    def _is_success(self, instance_data):
        return (instance_data['status'] == 'completed'
                and instance_data['result'] == 'success')

    def _block_sorted_func(self, item):
        """
        Логика тут такая - у кубиков композитора в отдельно взятом инстансе
        один и тот же code (но мы его не знаем) и поэтому нирвана
        добавляем $<int> в конец code чтобы их отличать

        Нам нужно отсортировать по возрастанию <int> и найти последний
        таким образом, результат которого нам и нужен
        """
        split_result = item['blockCode'].split('$')
        if len(split_result) != 2:
            return 0
        return int(split_result[-1])

    def _get_json_from_mds(self, mds_path):
        mds_client = MdsS3ApiClient()
        object_data = mds_client.get_object_content(mds_path)
        if object_data:
            return json.loads(object_data)

    def _get_instance_result(self, instance_data):
        if self._is_success(instance_data):
            block_results = nirvana_client.get_block_results(
                workflow_instance_id=self.pk,
                outputs=['result'],
            )
            block_results.sort(key=self._block_sorted_func)
            setattr(instance_data, 'output', self._get_json_from_mds(
                mds_path=block_results[-1]['results'][0]['directStoragePath'],
            ))

    def get_object(self):
        summary = nirvana_client.get_workflow_summary(
            workflow_instance_id=self.pk
        )
        self._get_instance_result(summary)
        return summary


class InstanceRestartApiView(NirvanaErrorHandlerMixin, APIView):
    def post(self, request, *args, **kwargs):
        workflow_instance_id = kwargs['pk']

        execution_state = nirvana_client.get_execution_state(workflow_instance_id=workflow_instance_id)
        if execution_state['result'] != 'failure':
            raise IllegalWorkflowInstanceStatus(non_field_messages='Workflow instance result not equal \'failure\'')

        cloned_workflow_instance_id = nirvana_client.clone_workflow_instance(workflow_instance_id=workflow_instance_id)
        nirvana_client.edit_workflow(workflow_instance_id=cloned_workflow_instance_id, execution_params={
            'resultCloningPolicy': 'deep',
        })

        nirvana_client.start_workflow(workflow_instance_id=cloned_workflow_instance_id)

        return Response({
            'workflow_instance_id': cloned_workflow_instance_id,
        })
