import json

from rest_framework.response import Response
from django.conf import settings
from . import base
from .base import APIView
from ..errors import WorkflowNotFoundError
from ..serializers.workflow import (
    WorkflowListSerializer,
    WorkflowDetailSerializer,
    WorkflowAdvertisingCampaignDataSerializer,
    WorkflowSiteTestingDataSerializer,
)
from ..utils.workflow import get_workflow_id
from ...clients.nirvana import nirvana_client


class WorkflowListApiView(base.ListAPIView):
    """
    Возвращает все воркфлоу проекта в нирване
    https://nirvana.yandex-team.ru/project/compositor
    """
    serializer_class = WorkflowListSerializer

    def get_queryset(self):
        return nirvana_client.find_workflows(
            pattern='',
            workflow_filters={'quotaProjectId': 'compositor'},
            pagination_data={'pageSize': 5000, 'pageNumber': 1},
        )

    def filter_queryset(self, queryset):
        """
        Апи нирваны возвращает все запуски воркфлоу, нужно
        отфильтровать уникальные
        """
        filtered_queryset = {}
        for obj in queryset:
            if obj.guid not in filtered_queryset:
                filtered_queryset[obj.guid] = obj
        return filtered_queryset.values()


class WorkflowDetailApiView(base.RetrieveAPIView):
    """
    Возвращает информацию по конкретному воркфлоу
    """
    serializer_class = WorkflowDetailSerializer
    pk_field = 'workflow_slug'

    def get_object(self):
        workflow_slug = self.pk
        workflow_id = get_workflow_id(workflow_slug)
        if workflow_id is None:
            raise WorkflowNotFoundError()

        metadata = nirvana_client.get_workflow_meta_data(
            workflow_id=workflow_id
        ).__dict__
        blocks_data = nirvana_client.get_workflow(
            workflow_id=workflow_id
        )['blocks']
        metadata['blocks'] = blocks_data

        return metadata


class WorkflowStartApiView(APIView):
    def post(self, request, *args, **kwargs):
        workflow_slug = kwargs['workflow_slug']
        workflow_id = get_workflow_id(workflow_slug)
        if workflow_id is None:
            raise WorkflowNotFoundError()

        data = request.data
        serializer = self.get_serializer(workflow_slug=workflow_slug, data=data)
        serializer.is_valid(raise_exception=True)
        instance_id = nirvana_client.clone_workflow_instance(workflow_id=workflow_id)
        nirvana_client._request(
            'updateOperationBlocks',
            dict(
                workflowInstanceId=instance_id,
            ),
        )
        workflow_info = nirvana_client.get_workflow(workflow_instance_id=instance_id)
        input_data_block_guid = workflow_info['blocks'][0]['blockGuid']
        nirvana_client.set_block_parameters(
            workflow_instance_id=instance_id,
            block_patterns=[{
                'guid': input_data_block_guid,
            }],
            params=[{
                'parameter': 'input',
                'value': json.dumps(serializer.data),
            }],
        )
        nirvana_client.start_workflow(workflow_instance_id=instance_id)

        return Response({
            'workflow_instance_id': instance_id,
        })

    def get_serializer(self, *args, **kwargs):
        workflow_slug = kwargs['workflow_slug']
        serializers = {
            settings.WORKFLOW_ADVERTISING_CAMPAIGN_SLUG: WorkflowAdvertisingCampaignDataSerializer,
            settings.WORKFLOW_SITE_TESTING_SLUG: WorkflowSiteTestingDataSerializer,
        }

        return serializers[workflow_slug](data=self.request.data, context={'request': self.request})
