import json
import requests
import logging

from requests.exceptions import HTTPError

from ..logic.tvm2_client import get_service_ticket
from .exceptions import InteractionException

logger = logging.getLogger(__name__)


class BaseClient:
    tvm_client = None
    base_exception = InteractionException

    @staticmethod
    def _log_response(request: dict, response: requests.Response):
        try:
            response_data = response.json()
        except ValueError:
            response_data = response.text

        logger.info(json.dumps({
            'request': request,
            'response': {
                'status_code': response.status_code,
                'headers': dict(response.headers),
                'data': response_data,
            },
        }))

    def _make_request(self, url, method='get', params=None, json_data=None, headers=None):
        response = getattr(requests, method)(
            url=url,
            params=params,
            json=json_data,
            headers=headers,
        )

        self._log_response(
            request={
                'url': url,
                'method': method,
            },
            response=response,
        )
        try:
            response.raise_for_status()
        except HTTPError as exc:
            raise self.base_exception(original_exception=exc)

        return response

    def _get_ticket(self):
        return get_service_ticket(self.tvm_client)
