import logging
from urllib.parse import urljoin

from .exceptions import ConnectInteractionException
from .. import settings
from .base import BaseClient

logger = logging.getLogger(__name__)


class ConnectClient(BaseClient):
    tvm_client = settings.DIRECTORY_TVM_CLIENT
    base_exception = ConnectInteractionException

    def __init__(self):
        self._host = settings.DIRECTORY_HOST

    def enable_service(self, org_id: int, author_id: str, service_slug: str):
        return self.toggle_service(org_id, author_id, service_slug, True)

    def disable_service(self, org_id: int, author_id: str, service_slug: str):
        return self.toggle_service(org_id, author_id, service_slug, False)

    def toggle_service(self, org_id: int, author_id: str, service_slug: str, position: bool):
        headers = {
            'X-Ya-Service-Ticket': self._get_ticket(),
            'X-ORG-ID': str(org_id),
            'X-UID': str(author_id),
            'X-USER-IP': '127.0.0.1',
        }

        if position:
            action = 'enable'
        else:
            action = 'disable'

        url = self._host + '/services/{slug}/{action}/'.format(slug=service_slug, action=action)

        response = self._make_request(url=url, method='post', headers=headers)

        if response.status_code != 201:
            raise self.base_exception()

        return response.json()

    def get_org(self, uid: str):
        headers = {
            'X-Ya-Service-Ticket': self._get_ticket(),
            'X-UID': str(uid),
            'X-USER-IP': '127.0.0.1',
        }
        url = urljoin(self._host, '/v11/organizations/')
        response = self._make_request(
            url=url,
            method='get',
            headers=headers,
        )
        return response.json()['result'][0]['id']

    def create_org(self, uid: str, name: str, language: str, domain_name: str):
        headers = {
            'X-Ya-Service-Ticket': self._get_ticket(),
            'X-UID': str(uid),
            'X-USER-IP': '127.0.0.1',
        }

        data = {
            'name': name,
            'language': language,
            'source': 'compositor',
        }

        url = self._host
        if domain_name:
            url = urljoin(url, '/organization/with-domain/')
            data['domain_name'] = domain_name
            data['tld'] = domain_name.split('.')[-1]
        else:
            url = urljoin(url, '/organization/without-domain/')
        try:
            response = self._make_request(
                url=url,
                method='post',
                json_data=data,
                headers=headers,
            )
        except self.base_exception as e:
            if e.original_exception.response.status_code == 409:
                # у пользователя уже есть организация
                # по текущей логики работы возвращаем ее
                return {'org_id': self.get_org(uid=uid)}
            raise

        if response.status_code != 201:
            raise self.base_exception()

        return response.json()
