import click
import json
import logging
import os

from functools import wraps

from .. import commands
from .utils import (
    validate_annotations,
    parse_input,
    get_func_params,
)

logger = logging.getLogger(__name__)


def nirvana_command(func):
    validate_annotations(func)

    def wrap_func(func):
        @click.command()
        @click.option("--input_file_path")
        @click.option("--output_file_path", required=False)
        @click.option("--tvm2_secret", required=False)
        @wraps(func)
        def wrapper(input_file_path, output_file_path=None, tvm2_secret=None):
            logger.info(f'Starting handling of {func.__name__}')
            if tvm2_secret:
                os.environ['TVM2_SECRET'] = tvm2_secret

            with open(input_file_path) as input_file:
                json_data = json.load(input_file)
                func_params = get_func_params(func)
                func_input = parse_input(json_data, func_params)

                logger.info('Calling intended function')

                data_to_write = func(**func_input)
                if data_to_write:
                    if not output_file_path:
                        message = (
                            'Receive result from function '
                            'but where is no file to write to'
                        )
                        raise RuntimeError(message)

                    # всегда обогащаем данные, которые пришли на вход и передаем их на выход
                    json_data.update(data_to_write)

                if output_file_path:
                    # Чтобы легче было собирать результат работы всегда на выход передаем
                    # данные, которые пришли на вход
                    with open(output_file_path, 'w') as output_file:
                        json.dump(json_data, output_file)

                logger.info('Successfully finished process')
        return wrapper

    click_command = wrap_func(func)
    command_name = f'{func.__name__}_command'
    click_command.__name__ = command_name
    if getattr(commands, command_name, None) is not None:
        raise RuntimeError(
            f'More then one attempt to set function with name {command_name}'
        )
    setattr(commands, command_name, click_command)

    return func
