import inspect
import logging
import urllib.parse

logger = logging.getLogger(__name__)


def parse_input(json_data: dict, func_params: dict) -> dict:
    """
    Проверяет наличие нужных ключей и их тип в json
    и подготавливает аргументы для передачи в функцию
    """
    logger.info('Validating input file')
    func_input = {}

    for arg, arg_data in func_params.items():
        input_arg = json_data.get(arg, inspect._empty)
        if input_arg is inspect._empty:
            if arg_data['default'] is inspect._empty:
                raise ValueError(
                    f'Value for {arg} was not provided and it does not have default'
                )
        else:
            arg_annotation = arg_data['annotation']
            if not isinstance(input_arg, arg_annotation):
                raise ValueError(
                    f'Expect {arg_annotation} for {arg} but got {type(input_arg)}'
                )
            if arg == 'website':
                # website обрабатываем по-особому оставляя только netloc
                input_arg = urllib.parse.urlparse(input_arg).netloc
            func_input[arg] = input_arg

    logger.info('Successfully validate input file')
    return func_input


def get_func_params(func):
    """
    Получает параметры функции и их данные
    """
    signature = inspect.signature(func)
    return {
        k: {
            'annotation': v.annotation,
            'default': v.default,
        }
        for k, v in signature.parameters.items()
    }


def validate_annotations(func):
    """
    Валидирует аннотации функции - у каждого параметра обязательно должна быть
    аннотация, а так же у функции должно быть указано возвратное значение
    либо None, либ dict
    """
    signature = inspect.signature(func)
    return_annotation = signature.return_annotation
    if return_annotation not in (dict, None):
        raise TypeError(f'Function should return None or Dict, got {return_annotation}')

    for arg, arg_data in signature.parameters.items():
        if arg_data.annotation is inspect._empty:
            raise SyntaxError(f'Every func argument should have annotation but {arg} does not')
