import logging
from copy import copy

from ids.registry import registry
from ids.exceptions import BackendError
from ids.services.startrek2.connector import ConflictMonster

from ..logic.tvm2_client import (
    get_service_ticket,
)
from ..logic.decorators import nirvana_command
from ..settings import (
    TRACKER_TVM_CLIENT,
    TRACKER_IDS_CONFIG,
    TRACKER_ISSUE_TYPE_CONFIG,
    TRACKER_RETRY,
)

logger = logging.getLogger(__name__)

TRACKER_DEFAULT_QUEUE_NAME = 'YANDEXTESTING'
TRACKER_DEFAULT_QUEUE_KEY = 'yandextesting'
TRACKER_DEFAULT_QUEUE_DESCRIPTION = 'Отчеты о тестировании сайтов'
TRACKER_TICKET_NAME_PATTERN = 'Тестирование {}'


def _get_tracker_repository(org_id, resource_type):
    repo_kwargs = copy(TRACKER_IDS_CONFIG)
    repo_kwargs['service_ticket'] = get_service_ticket(TRACKER_TVM_CLIENT)
    repo_kwargs['org_id'] = str(org_id)
    return registry.get_repository(
        resource_type=resource_type,
        **repo_kwargs
    )


@nirvana_command
def create_queue(uid: str, org_id: int, queue_name: str = None,
                 queue_key: str = None, queue_description: str = None) -> dict:
    """
    Функция создающая очередь в существующей организации с подключенным трекером
    """
    queue_name = queue_name or TRACKER_DEFAULT_QUEUE_NAME
    queue_key = queue_key or TRACKER_DEFAULT_QUEUE_KEY
    queue_description = queue_description or TRACKER_DEFAULT_QUEUE_DESCRIPTION

    queue_repository = _get_tracker_repository(
        org_id=org_id,
        resource_type='queues',
    )

    fields = {
        'name': queue_name,
        'key': queue_key,
        'lead': uid,
        'description': queue_description,
        'defaultType': 'task',
        'defaultPriority': 'normal',
        'issueTypesConfig': TRACKER_ISSUE_TYPE_CONFIG,
    }
    for _ in range(TRACKER_RETRY):
        try:
            queue_repository.create(
                params={
                    'uid': uid,
                    'allowReservedNames': True,
                },
                **fields
            )
            break
        except ConflictMonster:
            # Если уже есть такая очередь - все ок, делать ничего не нужно
            break
        except BackendError:
            logger.exception('Got exception while creating queue')
            continue
    else:
        raise BackendError('Retries ended')

    return {'queue_key': queue_key}


@nirvana_command
def create_ticket(uid: str, org_id: int, website: str, queue_key: str, ticket_name: str = None,
                  ticket_description: str = None) -> dict:
    """
    Функция создающая тикет в существующей организации с подключенным трекером
    """
    ticket_description = ticket_description or ''
    ticket_name = ticket_name or TRACKER_TICKET_NAME_PATTERN.format(website)

    ticket_repository = _get_tracker_repository(
        org_id=org_id, resource_type='issues',
    )
    fields = {
        'queue': queue_key,
        'summary': ticket_name,
        'description': ticket_description,
    }
    for _ in range(TRACKER_RETRY):
        try:
            ticket = ticket_repository.create(
                params={'uid': uid},
                **fields
            )
            break
        except BackendError:
            logger.exception('Got exception while creating ticket')
            continue
    else:
        raise BackendError('Retries ended')

    return {'ticket_key': ticket.key}
