from rest_framework.negotiation import DefaultContentNegotiation
from rest_framework.utils.mediatypes import _MediaType
from rest_framework import exceptions


class PriorityMediaType(_MediaType):
    def __init__(self, media_type_str):
        super(PriorityMediaType, self).__init__(media_type_str)
        self.quality = float(self.params.pop('q', '1.0'))

    def match(self, other):
        """Return true if this MediaType satisfies the given MediaType."""
        for key in self.params.keys():
            if other.params.get(key, None) != self.params.get(key, None):
                return False

        if self.sub_type != '*' and other.sub_type != '*' and other.sub_type != self.sub_type:
            return False

        if self.main_type != '*' and other.main_type != '*' and other.main_type != self.main_type:
            return False

        return True

    @property
    def precedence(self):
        return (
            self.main_type != '*',
            self.sub_type != '*',
            len(self.params) > 0,
            self.quality,
        )


class PriorityContentNegotiation(DefaultContentNegotiation):
    def select_renderer(self, request, renderers, format_suffix=None):
        """
        Given a request and a list of renderers, return a two-tuple of:
        (renderer, media type).
        """
        # Allow URL style format override.  eg. "?format=json
        format_query_param = self.settings.URL_FORMAT_OVERRIDE
        format = format_suffix or request.query_params.get(format_query_param)

        if format:
            renderers = self.filter_renderers(renderers, format)

        accepts = self.get_accept_list(request)
        accepts = list(map(PriorityMediaType, accepts))
        accepts.sort(key=lambda item: item.precedence, reverse=True)

        matches = []
        for renderer in renderers:
            renderer_media_type = PriorityMediaType(renderer.media_type)
            for media_type in accepts:
                if renderer_media_type.match(media_type):
                    matches.append((media_type, renderer))
                    break

        if matches:
            matches.sort(key=lambda pair: pair[0].precedence, reverse=True)
            media_type, renderer = matches[0]

            if PriorityMediaType(renderer.media_type).precedence > media_type.precedence:
                return renderer, renderer.media_type
            else:
                return renderer, media_type.orig

        raise exceptions.NotAcceptable(available_renderers=renderers)
