import re

from django.conf import settings
from rest_framework import serializers, fields

from intranet.crt.api.base.serializer_mixins.need_approve import NeedApproveCertMixin
from intranet.crt.constants import CA_NAME
from intranet.crt.core.ca import get_ca_cls
from intranet.crt.core.models import Host, Certificate
from intranet.crt.utils.ca import rewrite_certum_to_globalsign
from intranet.crt.utils.ssl import PemCertificateRequest
from intranet.crt.utils.domain import try_idna_decode


class HostsCertMixin(object):
    common_name_pattern = re.compile(r'^[^/=]+$')

    def create(self, validated_data):
        hosts = validated_data.pop('hosts')
        instance = Certificate(**validated_data)
        instance.save(hosts=hosts)
        for host in hosts:
            instance.hosts.add(host)
        return instance

    def postvalidate_hosts(self, attrs):
        hosts = [h.hostname for h in attrs.get('hosts', [])]

        if not hosts:
            common_name = attrs.get('common_name', None)
            if common_name:
                hosts = [common_name]
                attrs['hosts'] = [Host.objects.get_or_create(hostname=common_name)[0]]
            else:
                raise serializers.ValidationError('hosts должен содержать хотя бы один хост')

        if len(set(hosts)) < len(hosts):
            raise fields.ValidationError('hosts не может принимать дублирующиеся хосты')

        ca_class = get_ca_cls(attrs['ca_name'])
        non_whitelisted = ca_class.find_non_whitelisted_hosts(hosts)
        if non_whitelisted:
            raise serializers.ValidationError(
                'Хосты: {0} являются внешними и не могут быть выпущены этим УЦ '
                'автоматически. Обратитесь за дополнительной информацией через '
                'форму https://wiki.yandex-team.ru/security/ssl/'.format(', '.join(sorted(non_whitelisted))))

        non_auto = ca_class.find_non_auto_hosts(hosts)
        if non_auto:
            raise serializers.ValidationError(
                'Хосты: {0} не могут быть провалидированы этим УЦ '
                'автоматически. Обратитесь за дополнительной информацией через '
                'форму https://wiki.yandex-team.ru/security/ssl/'
                    .format(', '.join(non_auto)))

        wildcards = [host for host in hosts if '*' in host]
        if len(wildcards) > 1 and not ca_class.IS_SUPPORTING_MULTIPLE_WILDCARDS:
            raise serializers.ValidationError('Выбранный УЦ не поддерживает несколько wildcard.')

        return attrs

    def postvalidate_common_name_in_sans(self, attrs):
        if attrs['common_name'] not in {host.hostname for host in attrs['hosts']}:
            raise serializers.ValidationError('Common Name must be included in Certificate Subject Alternative Names (hosts)')

    def validate_request(self, csr):
        if not csr:
            return csr

        self.get_common_name(csr)  # Проверяем, что common_name парсится корректно

        return csr

    def validate_desired_ttl_days(self, value):
        if value is not None:
            value = min(value, settings.CRT_HOST_CERTS_MAX_TTL)
        return value

    def postvalidate_request(self, attrs):
        csr = attrs.get('request')
        if not csr:
            return attrs

        pem_certificate_request = PemCertificateRequest(csr)
        hosts = pem_certificate_request.sans
        common_name = self.get_common_name(csr)
        if not hosts:
            # если в CSR нет SAN, то список хостов
            # все равно должен содержать хотя бы тот хост, что в CN
            hosts = [common_name]

        if pem_certificate_request.email_address is not None:
            attrs['email'] = pem_certificate_request.email_address
        attrs['common_name'] = common_name
        attrs['hosts'] = [Host.objects.get_or_create(hostname=hostname)[0] for hostname in hosts]

        return attrs

    def rewrite_ca_name(self, attrs):
        ca_name = attrs.get('ca_name')
        if ca_name:
            attrs['ca_name'] = rewrite_certum_to_globalsign(ca_name)
        return attrs

    def decode_idna_encoded_initial(self):
        # Иногда нам могут заслать 'xn--d1acpjx3f.xn--p1ai'
        # желательно распознать это и не упасть
        requested_hosts = self.initial_data.get('hosts')
        if requested_hosts:
            idna_decoded_hosts = []
            for fqdn in requested_hosts.split(','):
                decoded_fqdn = try_idna_decode(fqdn)
                if decoded_fqdn not in idna_decoded_hosts:
                    idna_decoded_hosts.append(decoded_fqdn)
            self.initial_data['hosts'] = ','.join(idna_decoded_hosts)

        requested_common_name = self.initial_data.get('common_name')
        if requested_common_name:
            self.initial_data['common_name'] = try_idna_decode(requested_common_name)

    def filter_suspended_domains(self, attrs):
        common_name = attrs.get('common_name')
        if common_name and common_name.endswith(settings.CRT_SUSPENDED_DOMAINS):
            attrs['common_name'] = None

        hosts = attrs.get('hosts')
        if hosts:
            hosts = [h for h in hosts if not h.hostname.endswith(settings.CRT_SUSPENDED_DOMAINS)]
            attrs['hosts'] = hosts

        return attrs

    def prevalidate(self, attrs):
        attrs = self.rewrite_ca_name(attrs)
        attrs = self.filter_suspended_domains(attrs)
        attrs = self.postvalidate_request(attrs)
        attrs = self.postvalidate_hosts(attrs)

        return attrs

    def pre_save(self):
        super(HostsCertMixin, self).pre_save()

        if not self.validated_data.get('email'):
            self.validated_data['email'] = 'pki@yandex-team.ru'
        self.validated_data['requested_by_csr'] = 'request' in self.validated_data

    def is_valid(self, *args, **kwargs):
        self.decode_idna_encoded_initial()
        super().is_valid(*args, **kwargs)

    def validate(self, attrs):
        attrs = super(HostsCertMixin, self).validate(attrs)

        attrs = self.prevalidate(attrs)

        is_extended_validation = attrs.get('extended_validation', False)
        is_wildcard = any('*' in host.hostname for host in attrs['hosts'])

        if is_extended_validation and is_wildcard:
            raise serializers.ValidationError('EV certificate can not have wildcard certificates.')

        hosts = [h.hostname for h in attrs['hosts']]
        wildcards = [host for host in hosts if host.startswith('*.')]

        common_name = attrs.get('common_name')
        if common_name is None:
            common_name = wildcards[0] if wildcards else hosts[0]

        if attrs['ca_name'] in CA_NAME.GLOBALSIGN_CAS:
            if wildcards and not common_name.startswith('*.'):
                common_name = wildcards[0]

        cn_match = self.common_name_pattern.match(common_name)
        if cn_match is None:
            raise serializers.ValidationError('Invalid common name')

        if len(common_name) > 255:
            raise serializers.ValidationError(
                'Length of certificate CN=({}) must by less or eq 255 symbols'
                    .format(common_name)
            )
        attrs['common_name'] = common_name
        self.postvalidate_common_name_in_sans(attrs)

        return attrs

    def to_representation(self, instance):
        result = super(HostsCertMixin, self).to_representation(instance)
        if 'hosts' in result:
            result['hosts'] = [host['hostname'] for host in result['hosts']]
        return result


class ExternalHostsCertMixin(NeedApproveCertMixin, HostsCertMixin):
    pass


class HostsRequiredCertMixin(object):
    def check_required_hosts(self, hosts):
        if not hosts:
            raise serializers.ValidationError('Hosts must contain at least one host')

        if len(set(hosts)) < len(hosts):
            raise serializers.ValidationError('Hosts must not duplicate each other')

    def fill_common_name_if_not_set(self, attrs):
        if not attrs.get('common_name'):
            attrs['common_name'] = attrs['hosts'][0].hostname

    def fill_hosts_with_common_name(self, attrs):
        host, _ = Host.objects.get_or_create(hostname=attrs['common_name'])
        attrs['hosts'] = [host]

    def validate(self, attrs):
        attrs = self.check_all_certs_validation(attrs)
        if self.errors:
            raise serializers.ValidationError(self.errors)

        if not attrs.get('hosts'):
            if not attrs.get('common_name'):
                raise serializers.ValidationError('Either hosts or common_name must be specified')
            attrs = self.postvalidate_common_name(attrs)
        attrs = self.postvalidate_hosts(attrs)
        self.postvalidate_common_name_in_sans(attrs)

        return attrs
