import re
from rest_framework import serializers
from django.utils.translation import ugettext_lazy as _

from intranet.crt.core.models import AssessorCertificate
from intranet.crt.core.utils import get_inum_by_sn
from intranet.crt.users.models import CrtUser


class SelfRequestOnlyMixin(object):
    """This mixin extracts user from common name and checks if it is equal to the current user.
    Extracting user from common name is possible iff it "looks like a common name of the certificate
    for user", i.e. the requested certificate is not requested for a domain or some hardware.
    This looks hacky, I hope when we update to DRF3 we'll find a better solution.
    """

    def validate_common_name(self, common_name):
        if not common_name:
            return common_name

        user = self.get_user_from_common_name(common_name)

        if self.context['request'].user != user:
            raise serializers.ValidationError(_('Invalid user in common name'))

        return common_name


class NinjaCertMixin(SelfRequestOnlyMixin):
    common_name_pattern = re.compile(r'^(?!robot-)([\w._-]+)@pda-ld\.yandex\.ru$')


class NinjaExchangeCertMixin(SelfRequestOnlyMixin):
    common_name_pattern = re.compile(r'^(?!robot-)([\w._-]+)@ld\.yandex\.ru$')


class HypercubeCertMixin(NinjaCertMixin):
    def validate_common_name(self, common_name):
        if not common_name:
            return common_name
        self.get_user_from_common_name(common_name)
        return common_name

    def validate(self, attrs):
        attrs = super(HypercubeCertMixin, self).validate(attrs)

        common_name = attrs.get('common_name', None)

        if common_name is None:
            raise serializers.ValidationError('Cannot find common_name field')

        user = self.get_user_from_common_name(common_name)
        if not (user.has_active_ld_certificates() or user.is_pdas_whitelisted()):
            raise serializers.ValidationError(f'User {user.username} has no active certificates '
                                              'to access Yandex network. Issuing prohibited.')
        attrs['email'] = user.email
        attrs['user'] = user

        return attrs

    def pre_save(self):
        self.validated_data['requester'] = self._context['request'].user
        self.validated_data['requested_by_csr'] = 'request' in self.validated_data


class TempPcCertMixin(object):
    common_name_pattern = re.compile(r'^([\w._-]+)@pda-ld\.yandex\.ru$')

    def validate_requestee(self, requestee):
        requester = self.context['request'].user
        if requester != requestee:
            raise serializers.ValidationError('CSR Common Name does not match authorized user')

    def validate_request(self, csr):
        if csr is None:
            return csr

        common_name = self.get_common_name(csr)
        user = self.get_user_from_common_name(common_name)
        self.validate_requestee(user)

        if not (user.has_active_ld_certificates() or user.is_pdas_whitelisted()):
            raise serializers.ValidationError(f'User {user.username} has no active certificates '
                                              'to access Yandex network. Issuing prohibited.')

        return csr

    def pre_save(self):
        if not self.is_valid():
            raise

        self.validated_data['requester'] = self._context['request'].user
        self.validated_data['email'] = self._context['request'].user.email
        self.validated_data['requested_by_csr'] = 'request' in self.validated_data


class PcCertMixin(object):
    def validate_requestee(self, requestee):
        requester = self.context['request'].user
        if requester != requestee and not (
            requester.has_perm('core.can_issue_device_certificates')
            or (
                requester.has_perm('core.can_issue_device_certificates_for_external')
                and
                requestee.is_external_employee()
            )
        ):
            raise serializers.ValidationError('Invalid user in common name')

    def validate_common_name(self, common_name):
        if common_name is None:
            return common_name

        user = self.get_user_from_common_name(common_name)
        self.validate_requestee(user)

        return common_name

    def validate_request(self, csr):
        if csr is None:
            return csr

        common_name = self.get_common_name(csr)
        user = self.get_user_from_common_name(common_name)
        self.validate_requestee(user)
        self.get_email(csr)

        return csr

    def prevalidate_pc_inum(self, attrs):
        pc_inum = attrs.get('pc_inum')
        if not pc_inum:
            attrs['pc_inum'] = get_inum_by_sn(attrs.get('pc_serial_number'))
        return attrs

    def validate(self, attrs):
        attrs = super(PcCertMixin, self).validate(attrs)

        attrs = self.prevalidate_pc_inum(attrs)

        common_name = None
        # Сделаем вид, что в common_name ничего не передавали, если там пустая строка или null
        if attrs.get('common_name'):
            if 'request' in attrs:
                raise serializers.ValidationError('Both common_name and request given.')
            common_name = attrs['common_name']

        elif 'request' in attrs:
            csr = attrs['request']
            common_name = self.get_common_name(csr)

        if common_name is None:
            raise serializers.ValidationError('Cannot find common_name/request field')

        user = self.get_user_from_common_name(common_name)
        attrs['email'] = user.email
        attrs['user'] = user

        return attrs

    def pre_save(self):
        self.validated_data['requester'] = self._context['request'].user
        self.validated_data['requested_by_csr'] = 'request' in self.validated_data


class BankPcCertMixin(PcCertMixin):
    def validate_request(self, csr):
        if not csr:
            raise serializers.ValidationError('Field must contain valid CSR')

        common_name = self.get_common_name(csr)
        self.get_user_from_common_name(common_name)
        self.get_email(csr)

        return csr


class ZombPcCertMixin(object):
    common_name_pattern = re.compile(r'^[A-Za-z0-9.-]+\.(zombie|wst)\.yandex\.net$')
    st_id_re = re.compile(r'^[A-Za-z]+-\d+$')

    def validate_request(self, csr):
        if csr is None:
            return csr
        self.get_common_name(csr)
        return csr

    def validate(self, attrs):
        attrs = super(ZombPcCertMixin, self).validate(attrs)
        attrs = self.prevalidate_pc_inum(attrs)
        if self.st_id_re.match(attrs['hardware_request_st_id']) is None:
            raise serializers.ValidationError({'hardware_request_st_id': 'st-id should be a valid tracker key'})
        return attrs

    def prevalidate_pc_inum(self, attrs):
        pc_inum = attrs.get('pc_inum')
        if not pc_inum:
            attrs['pc_inum'] = get_inum_by_sn(attrs.get('pc_serial_number'))
        return attrs

    def pre_save(self):
        self.validated_data['requester'] = self.validated_data['user'] = self._context['request'].user
        self.validated_data['requested_by_csr'] = 'request' in self.validated_data


class AssessorCertMixin(PcCertMixin):
    common_name_pattern = re.compile(r'^([\w._-]+)@assessors\.yandex-team\.ru$')

    @classmethod
    def validate_common_name(cls, username):
        try:
            user = CrtUser.objects.get(username=username, is_active=True)
        except CrtUser.DoesNotExist:
            raise serializers.ValidationError('Пользователь не существует либо уволен')

        if not AssessorCertificate.objects.can_issue_new(user):
            raise serializers.ValidationError('Can not issue new certificate for this assessor')

        return '{0}@assessors.yandex-team.ru'.format(username)
