import abc
import datetime

from django.conf import settings
from django.utils import timezone
from rest_framework import exceptions, serializers

from intranet.crt.api.v1.certificates.serializers.specified import VpnTokenCertSerializer
from intranet.crt.core.utils import get_inums_and_models
from intranet.crt.exceptions import CrtBotNotFound
from intranet.crt.utils.log import security_log
from intranet.crt.utils.ssl import PemCertificateRequest


class DesiredTTLDaysSerializer(serializers.Serializer):
    desired_ttl_days = serializers.IntegerField(min_value=1, required=False)


class BaseReissueValidator(object, metaclass=abc.ABCMeta):
    @abc.abstractproperty
    def old_cert_ttl_threshold_days(self):
        pass

    old_cert_revoke_days = 5

    def __init__(self, request):
        self.request = request

    def _is_set_reuse_csr(self, old_cert, reuse_required=True):
        reuse_csr = self.request.data.get('reuse_csr')
        if reuse_csr is not True:
            if reuse_required:
                raise exceptions.ParseError('{} certificates can only be reissued with old CSR reusing'
                                            .format(old_cert.type.name))
            else:
                return False

        if old_cert.request is None:
            raise exceptions.ParseError('This {} certificate has no CSR saved and can\'t be reissued '
                                        'with old CSR reusing'.format(old_cert.type.name))

        return True

    def _is_set_force(self, old_cert):
        force_mode = self.request.data.get('force')
        if force_mode is True:
            security_log(
                'User {} started force reissue for certificate {}'.format(self.request.user.username, old_cert.id)
            )
            return True

        return False

    def _validate_csr(self, client_cn):
        csr = self.request.data.get('request')
        if csr is None:
            raise exceptions.ParseError('"request" field was not found')

        if PemCertificateRequest(csr).common_name != client_cn:
            raise exceptions.ParseError('Old CN != new CN')

        return csr

    def _validate_threshold(self, old_cert):
        old_cert_ttl_threshold = datetime.timedelta(days=self.old_cert_ttl_threshold_days)
        if old_cert.end_date - timezone.now() >= old_cert_ttl_threshold:
            raise exceptions.ParseError(
                'Current certificate is not expiring. Threshold is {threshold} days'
                .format(threshold=self.old_cert_ttl_threshold_days)
            )
        return True

    def get_additional_fields(self, old_cert):
        return {}

    def get_desired_desired_ttl_days(self, old_cert):
        serializer = DesiredTTLDaysSerializer(data=self.request.data)
        serializer.is_valid(raise_exception=True)
        return serializer.validated_data.get('desired_ttl_days') or old_cert.desired_ttl_days


class PcReissueValidator(BaseReissueValidator):
    @property
    def old_cert_ttl_threshold_days(self):
        return settings.CRT_REISSUE_PC_THRESHOLD_DAYS

    def validate_pc_inum(self, cert_user):
        pc_inum = self.request.data.get('pc_inum')
        if not pc_inum:
            return None

        try:
            inums_and_models = get_inums_and_models(cert_user.username)
            if pc_inum not in {hw[0] for hw in inums_and_models}:
                raise exceptions.ParseError(f'User {cert_user.username} has no '
                                            f'device with pc_inum \'{pc_inum}\'')
        except CrtBotNotFound:
            # CERTOR-1979
            # некоторых внешних мы никогда не найдем в BOT
            pass

        return pc_inum

    def get_additional_fields(self, old_cert):
        pc_inum = self.validate_pc_inum(old_cert.get_reissue_user())

        fields = super(PcReissueValidator, self).get_additional_fields(old_cert)
        fields.update({
            'pc_hostname': self.request.data.get('pc_hostname') or old_cert.pc_hostname,
            'pc_os': self.request.data.get('pc_os') or old_cert.pc_os,
            'pc_serial_number': self.request.data.get('pc_serial_number') or old_cert.pc_serial_number,
            'pc_mac': self.request.data.get('pc_mac') or old_cert.pc_mac,
            'pc_inum': pc_inum or old_cert.pc_inum,
            'helpdesk_ticket': self.request.data.get('helpdesk_ticket') or old_cert.helpdesk_ticket,
        })

        if 'request' in self.request.data:
            csr = self._validate_csr(old_cert.common_name)
            fields['request'] = csr

        # TODO(lavrukov): Когда запретим дубли pc сертов необходимо будет здесь
        # реализовать проверку на дубли
        return fields


class RcServerReissueValidator(BaseReissueValidator):
    @property
    def old_cert_ttl_threshold_days(self):
        return settings.CRT_REISSUE_RC_THRESHOLD_DAYS

    def get_additional_fields(self, old_cert):
        force_mode = self._is_set_force(old_cert)
        if not force_mode:
            self._validate_threshold(old_cert)

        return super(RcServerReissueValidator, self).get_additional_fields(old_cert)


class YcServerReissueValidator(BaseReissueValidator):
    def old_cert_ttl_threshold_days(self):
        return settings.CRT_REISSUE_YC_THRESHOLD_DAYS

    def get_additional_fields(self, old_cert):
        force_mode = self._is_set_force(old_cert)
        if not force_mode:
            self._validate_threshold(old_cert)

        return super(YcServerReissueValidator, self).get_additional_fields(old_cert)


class VpnTokenReissueValidator(BaseReissueValidator):
    @property
    def old_cert_ttl_threshold_days(self):
        return settings.CRT_REISSUE_VPN_THRESHOLD_DAYS

    def get_additional_fields(self, old_cert):
        reuse_csr = self._is_set_reuse_csr(old_cert, reuse_required=True)
        force_mode = self.request.data.get('force')
        if not force_mode:
            self._validate_threshold(old_cert)

        fields = super(VpnTokenReissueValidator, self).get_additional_fields(old_cert)

        if reuse_csr:
            fields['request'] = old_cert.request
            fields['requested_by_csr'] = old_cert.requested_by_csr

        VpnTokenCertSerializer.validate_affiliation_and_hw(fields['request'])

        return fields


class ClientServerReissueValidator(BaseReissueValidator):
    @property
    def old_cert_ttl_threshold_days(self):
        return settings.CRT_REISSUE_CLIENT_SERVER_THRESHOLD_DAYS

    def get_additional_fields(self, old_cert):
        reuse_csr = self._is_set_reuse_csr(old_cert, reuse_required=False)
        force_mode = self._is_set_force(old_cert)
        if not force_mode:
            self._validate_threshold(old_cert)

        fields = super(ClientServerReissueValidator, self).get_additional_fields(old_cert)

        if reuse_csr:
            fields['request'] = old_cert.request
            fields['requested_by_csr'] = old_cert.requested_by_csr
            return fields

        if 'request' in self.request.data:
            client_csr = self._validate_csr(old_cert.common_name)
        else:
            raise exceptions.ParseError('CSR request is empty. It should be specified for '
                                        '{} certificate type'.format(old_cert.type.name))
        fields['request'] = client_csr

        return fields


class BankClientServerReissueValidator(BaseReissueValidator):
    @property
    def old_cert_ttl_threshold_days(self):
        return settings.CRT_REISSUE_BANK_CLIENT_SERVER_THRESHOLD_DAYS

    def get_additional_fields(self, old_cert):
        reuse_csr = self._is_set_reuse_csr(old_cert, reuse_required=False)
        force_mode = self._is_set_force(old_cert)
        if not force_mode:
            self._validate_threshold(old_cert)

        fields = super(BankClientServerReissueValidator, self).get_additional_fields(old_cert)

        if reuse_csr:
            fields['request'] = old_cert.request
            fields['requested_by_csr'] = old_cert.requested_by_csr
            return fields

        if 'request' in self.request.data:
            client_csr = self._validate_csr(old_cert.common_name)
        else:
            raise exceptions.ParseError('CSR request is empty. It should be specified for '
                                        '{} certificate type'.format(old_cert.type.name))
        fields['request'] = client_csr

        return fields


class TpmSmartcard1CReissueValidator(BaseReissueValidator):
    @property
    def old_cert_ttl_threshold_days(self):
        return settings.CRT_REISSUE_TPM_SMARTCARD_1C_THRESHOLD_DAYS

    def get_additional_fields(self, old_cert):
        reuse_csr = self._is_set_reuse_csr(old_cert, reuse_required=False)

        if not (reuse_csr or 'request' in self.request.data):
            raise exceptions.ParseError('Either \'request\' or \'reuse_csr\' should be passed '
                                        'for {} certificate type'.format(old_cert.type.name))

        if not self.request.data.get('force'):
            self._validate_threshold(old_cert)

        fields = super(TpmSmartcard1CReissueValidator, self).get_additional_fields(old_cert)

        fields['request'] = self.request.data.get('request')
        fields['requested_by_csr'] = True
        if reuse_csr:
            fields['request'] = old_cert.request
            fields['requested_by_csr'] = old_cert.requested_by_csr

        return fields
