from rest_framework import serializers

from intranet.crt.constants import AFFILIATION, CERT_TYPE


class VpnTokenCertMixin(object):
    @classmethod
    def validate_affiliation_and_hw(cls, csr):
        common_name = cls.get_common_name(csr)
        user = cls.get_user_from_common_name(common_name)

        if user.affiliation == AFFILIATION.YANDEX:
            raise serializers.ValidationError(
                f'{CERT_TYPE.VPN_TOKEN} certificates is only for external users.'
            )
        if user.has_bot_hardware():
            raise serializers.ValidationError(
                f'{CERT_TYPE.VPN_TOKEN} certificates is only for users without BOT hardware.'
            )
        if user.certificates.active_user_hardware_types().exists():
            raise serializers.ValidationError(
                f'{CERT_TYPE.VPN_TOKEN} certificates is only for users without personal hardware.'
            )

    def is_permitted_self_request(self, requester, user):
        return requester == user and requester.has_perm('core.can_issue_vpn_token_certificates_for_self')

    def validate_request(self, csr):
        if csr is None:
            raise serializers.ValidationError('This field is required.')
        return csr

    def validate(self, attrs):
        attrs = super().validate(attrs)

        requester = self._context['request'].user
        common_name = self.get_common_name(attrs['request'])
        user = self.get_user_from_common_name(common_name)
        attrs['user'] = user

        if not self.is_permitted_self_request(requester, user):
            self.validate_affiliation_and_hw(attrs['request'])

        return attrs

    def pre_save(self):
        self.validated_data['requester'] = self._context['request'].user
        self.validated_data['requested_by_csr'] = 'request' in self.validated_data
