from django.conf import settings
from rest_framework import views, response, reverse, exceptions
from rest_framework.fields import BooleanField, NullBooleanField

from intranet.crt.api.v2.misc.fields import SuggestableField
from intranet.crt.api.v2.certificates.serializers.dispatch import get_serializer
from intranet.crt.core.ca import get_ca


class RequestFieldsView(views.APIView):
    DEFAULT_FIELDS = {'ca_name', 'type', 'yav_secret_id', 'notify_on_expiration'}  # не отдаём их в ручке
    FIELD_NAMES = {
        'abc_service': {'ru': 'ABC-сервис', 'en': 'ABC service'},
        'desired_ttl_days': {'ru': 'Желаемый TTL (в днях)', 'en': 'Desired TTL (days)'},
        'extended_validation': {'ru': 'Extended Validation', 'en': 'Extended Validation'},
        'hosts': {'ru': 'Хосты', 'en': 'Hosts'},
        'request': {'ru': 'CSR', 'en': 'CSR'},
        'pc_hostname': {'ru': 'Имя PC', 'en': 'PC Hostname'},
        'pc_inum': {'ru': 'Инвентарный номер', 'en': 'Inventory Number'},
        'pc_mac': {'ru': 'MAC-адреса', 'en': 'MAC-addresses'},
        'pc_os': {'ru': 'Операционная система', 'en': 'Operational System'},
        'pc_serial_number': {'ru': 'Серийный номер PC', 'en': 'PC serial number'},
        'common_name': {'ru': 'Common name', 'en': 'Common name'},
        'user': {'ru': 'Пользователь', 'en': 'User'},
        'is_ecc': {'ru': 'ECC сертификат', 'en': 'ECC certificate'},
        'hardware_request_st_id': {'ru': 'Ключ тикета на запрос тачки', 'en': 'Ticket key'},
        'helpdesk_ticket': {'ru': 'Ключ тикета helpdesk', 'en': 'Helpdesk ticket key'}
    }

    @classmethod
    def expand_fields(cls, fields, required_fields):
        values = []
        for name, field in fields.items():
            suggest_path = ''
            field_type = 'textinput'
            if isinstance(field, BooleanField) or isinstance(field, NullBooleanField):
                field_type = 'bool'
            elif isinstance(field, SuggestableField):
                field_type = 'suggest'
                suggest_path = reverse.reverse(field.suggest_path)

            value = {
                'slug': name,
                'title': {
                    'ru': cls.FIELD_NAMES[name]['ru'],
                    'en': cls.FIELD_NAMES[name]['en'],
                },
                'type': field_type,
                'suggestPath': suggest_path,
                'isRequired': name in required_fields,
            }
            values.append(value)

        return values

    @classmethod
    def get_fields(cls, request):
        ca_name = request.query_params.get('ca_name')
        if ca_name not in settings.AVAILABLE_CA:
            raise exceptions.ParseError('Invalid ca_name parameter')

        ca = get_ca(ca_name)
        cert_type = request.query_params.get('type')
        if cert_type not in ca.supported_types:
            raise exceptions.ParseError('Invalid type parameter')

        serializer = get_serializer(cert_type, ca_name)()
        readonly_fields = set(getattr(serializer.Meta, 'read_only_fields', []))
        required_fields = set(getattr(serializer.Meta, 'required', [])) - readonly_fields
        fields = set(serializer.Meta.fields) - readonly_fields

        fields = {field: serializer.fields[field] for field in fields}

        for name, field in list(fields.items()):
            if field.read_only:
                del fields[name]
                required_fields.discard(name)
                continue

            if field.required:
                required_fields.add(name)

        for field in cls.DEFAULT_FIELDS:
            fields.pop(field, None)
        required_fields -= cls.DEFAULT_FIELDS

        # CERTOR-851: поле с CSR должно отсутствовать в форме
        fields.pop('request', None)
        required_fields.discard('request')

        return fields, required_fields

    def get(self, request):
        fields, required = self.get_fields(request)
        expanded_fields = self.expand_fields(fields, required)

        return response.Response(expanded_fields)
