import abc
from collections import OrderedDict
from rest_framework import views, pagination, response


class SuggestLimitOffsetPagination(pagination.LimitOffsetPagination):
    def get_paginated_response(self, data, request, full_result_passed=False):
        limit = self.get_limit(request)
        offset = self.get_offset(request)
        if full_result_passed:
            data = data[offset:offset + limit]

        return response.Response(OrderedDict([
            ('limit', limit),
            ('offset', offset),
            ('results', data),
        ]))


class SuggestPaginatedView(views.APIView):
    pagination_class = SuggestLimitOffsetPagination

    @property
    def paginator(self):
        if not hasattr(self, '_paginator'):
            if self.pagination_class is None:
                self._paginator = None
            else:
                self._paginator = self.pagination_class()
        return self._paginator


class ConstantSuggestPaginatedView(SuggestPaginatedView, metaclass=abc.ABCMeta):
    @abc.abstractmethod
    def prepare_object(self, constant):
        pass

    @abc.abstractmethod
    def get_constants(self):
        pass

    def get(self, request):
        constants = [self.prepare_object(const) for const in self.get_constants()]

        slugs = request.query_params.getlist('id', [])
        search = request.query_params.get('search')

        if slugs:
            constants = [const for const in constants if const['id'] in slugs]
        elif search:
            search = search.lower()
            constants = [
                const for const in constants if
                (
                        search in const['id'] or
                        search in const['title']['en'].lower() or
                        search in const['title']['ru'].lower()
                )]

        return self.paginator.get_paginated_response(constants, request, full_result_passed=True)
