from collections import OrderedDict
from django.conf import settings
from intrasearch_fetcher import IntrasearchFetcher
from intrasearch_fetcher.exceptions import IntrasearchFetcherException
from rest_framework import exceptions

from intranet.crt.api.frontend.suggest.base import SuggestPaginatedView
from intranet.crt.users.models import CrtUser


class UserSuggestView(SuggestPaginatedView):
    @staticmethod
    def search(search, limit, offset):
        fetcher = IntrasearchFetcher(
            'people',
            settings.CRT_IDS_USER_AGENT,
            settings.CRT_OAUTH_TOKEN,
            settings.CRT_INTRASEARCH_TIMEOUT,
        )
        try:
            return fetcher.search(
                search,
                filters={},
                language='en',
                limit=limit,
                offset=offset,
            )
        except IntrasearchFetcherException as err:
            raise exceptions.APIException(code=500, detail=err.args)

    @staticmethod
    def prepare_intrasearch_object(value):
        return {
            'id': value['id'],
            'title': {
                'en': value['title'],
                'ru': None,
            },
            'type': 'default',
        }

    @staticmethod
    def prepare_internal_object(value):
        return {
            'id': value.username,
            'title': {
                'en': value.full_name,
                'ru': value.full_name_ru,
            },
            'type': 'default',
        }

    @staticmethod
    def fill_ru_names(results):
        results_dict = OrderedDict([(obj['id'], obj) for obj in results])

        query = CrtUser.objects.filter(username__in=results_dict.keys())
        for obj in query:
            results_dict[obj.username]['title']['ru'] = obj.full_name_ru

        return list(results_dict.values())

    def get(self, request):
        usernames = request.query_params.getlist('id', [])
        if usernames:
            query = CrtUser.objects.filter(username__in=usernames)
            query = self.paginator.paginate_queryset(query, request)
            result = [self.prepare_internal_object(obj) for obj in query]
            return self.paginator.get_paginated_response(result, request)

        search = request.query_params.get('search', '')
        limit = self.paginator.get_limit(request)
        offset = self.paginator.get_offset(request)

        raw_results = self.search(search, limit, offset)
        results = [self.prepare_intrasearch_object(value) for value in raw_results]
        results = self.fill_ru_names(results)
        return self.paginator.get_paginated_response(results, request)

