from django.db import transaction
from rest_framework import serializers, fields

from intranet.crt.core.models import ApproveRequest
from intranet.crt.utils.startrek import close_st_issue_for_cert_approval


class ApproveRequestSerializer(serializers.ModelSerializer):
    approver = fields.ReadOnlyField(source='approver.username')
    cert_status = fields.ReadOnlyField(source='certificate.status')

    def validate(self, attrs):
        if getattr(self, 'instance', None) and self.instance.approver is not None:
            raise serializers.ValidationError('Данный запрос уже обработан')
        return super(ApproveRequestSerializer, self).validate(attrs)

    @transaction.atomic
    def save(self, **kwargs):
        self.instance.approver = self.context['request'].user
        cert = self.instance.certificate

        if self.validated_data['approved']:
            comment = 'Request approved'
            cert.controller.approve_request(user=self.instance.approver)
        else:
            comment = 'Request rejected'
            cert.controller.reject_request(user=self.instance.approver)

        result = super(ApproveRequestSerializer, self).save(**kwargs)

        close_st_issue_for_cert_approval(cert, comment=comment, approver=self.context['request'].user)

        return result

    class Meta:
        model = ApproveRequest
        fields = ('approved', 'approver', 'cert_status')
