import django_filters
from django import forms
from django.conf import settings
from django.db.models import Q
from django_abc_data.models import AbcService

from intranet.crt.core.models import Certificate, CertificateType
from intranet.crt.constants import CERT_STATUS
from intranet.crt.core.utils import expand_wildcard_patterns

available_cas = [(ca_name, ca_name) for ca_name in settings.AVAILABLE_CA]

available_cas_uppercase_normalcase = {ca_name.upper(): ca_name for ca_name, _ in available_cas}


class HostField(forms.MultipleChoiceField):
    # игнорируем self.choices
    def valid_value(self, value):
        return True


class HostFilter(django_filters.CharFilter):
    field_class = HostField

    def __init__(self, *args, **kwargs):
        super(HostFilter, self).__init__(*args, **kwargs)
        self.field_name = 'hosts__hostname'
        self.lookup_expr = 'in'

    def filter(self, qs, value):
        patterns = set()

        for fqdn in value:
            patterns.update(expand_wildcard_patterns(fqdn))

        if patterns:
            return super(HostFilter, self).filter(qs, patterns)
        else:
            return qs


class CertificateFilter(django_filters.FilterSet):
    id = django_filters.CharFilter()
    id__gt = django_filters.NumberFilter(field_name='id', lookup_expr='gt')
    id__lt = django_filters.NumberFilter(field_name='id', lookup_expr='lt')
    tag = django_filters.CharFilter(field_name='tags__name')
    status = django_filters.MultipleChoiceFilter(choices=CERT_STATUS.choices(), distinct=False)
    ca_name = django_filters.ChoiceFilter(choices=available_cas, lookup_expr='exact')
    type = django_filters.ModelMultipleChoiceFilter(
        field_name='type__name',
        to_field_name='name',
        queryset=CertificateType.objects.all(),
        distinct=False,
    )
    username = django_filters.CharFilter(method='filter_username')
    user = username  # username - для v1, user - для v2

    requester = django_filters.CharFilter(field_name='requester__username')
    serial_number = django_filters.CharFilter()
    common_name = django_filters.CharFilter()
    common_name__like = django_filters.CharFilter(method='filter_common_name__like')
    host__exact = django_filters.CharFilter(field_name='hosts__hostname')
    host = HostFilter()
    pc_hostname = django_filters.CharFilter()
    pc_os = django_filters.CharFilter()
    pc_serial_number = django_filters.CharFilter()
    pc_mac = django_filters.CharFilter()
    pc_inum = django_filters.CharFilter()
    abc_service = django_filters.ModelMultipleChoiceFilter(
        field_name='abc_service__external_id',
        to_field_name='external_id',
        lookup_expr='exact',
        queryset=AbcService.objects.all(),
        distinct=False,
    )
    abc_service__isnull = django_filters.BooleanFilter(method='filter_abc_service__is_null')
    updated__gt = django_filters.IsoDateTimeFilter(field_name='updated', lookup_expr='gt')

    def filter_abc_service__is_null(self, queryset, name, value):
        return queryset.filter(abc_service__isnull=value)

    def filter_common_name__like(self, queryset, name, value):
        return queryset.filter(Q(common_name__icontains=value) | Q(hosts__hostname__icontains=value))

    def filter_username(self, queryset, name, value):
        value_set = {value.strip() for value in value.split(',') if value.strip()}
        if len(value_set) > 1:
            return queryset.filter(user__username__in=value_set)
        return queryset.filter(user__username=value_set.pop())

    class Meta:
        model = Certificate
        fields = [
            'id',
            'type',
            'status',
            'serial_number',
            'username',
            'user',
            'requester',
            'host__exact',
            'common_name__like',
            'pc_hostname',
            'pc_os',
            'pc_serial_number',
            'pc_mac',
            'pc_inum',
            'tag',
            'host',
            'ca_name',
            'abc_service',
            'abc_service__isnull',
        ]
