from django.conf import settings
from django.utils.translation import ugettext_lazy
from django.utils.encoding import force_text
from rest_framework import permissions

from intranet.crt.api.base.exceptions import BadRequest
from intranet.crt.core.ca import get_ca_cls
from intranet.crt.constants import (
    ABC_ADMINISTRATOR_SCOPE, ABC_CERTIFICATE_MANAGER_SCOPE, CERT_TYPE, CERT_DETAIL_ACTION, CERT_EXTENSION
)
from intranet.crt.utils.ssl import (
    request_contains_custom_crt_extensions, get_x509_custom_extensions, PemCertificateRequest
)
from intranet.crt.api.v1.certificates.serializers.specified import ZombieCertSerializer


class CertificateActionsPermission(permissions.BasePermission):
    @staticmethod
    def check_is_owner(request, view, obj):
        return IsOwner().has_object_permission(request, view, obj)

    def check_revoke_perm(self, request, view, obj):
        if request.user.has_perm('core.can_revoke_any_certificate'):
            return True
        if request.user.has_perm('core.can_revoke_users_certificates') and obj.type.name in CERT_TYPE.USERS_TYPES:
            return True
        return self.check_is_owner(request, view, obj)

    def check_hold(self, request, view, obj):
        return (
            request.user.has_perm('core.can_revoke_any_certificate')
            or (request.user.has_perm('core.can_revoke_users_certificates') and obj.type.name in CERT_TYPE.USERS_TYPES)
        )

    def check_update(self, request, view, obj):
        is_owner = self.check_is_owner(request, view, obj)

        if 'abc_service' in request.data:
            if not request.user.has_perm('core.can_change_abc_service_for_any_certificate') and not is_owner:
                return False
        if 'manual_tags' in request.data:
            if not request.user.has_perm('core.can_edit_certificate_tags'):
                return False

        return True

    def check_revoke(self, request, view, obj):
        return self.check_revoke_perm(request, view, obj)

    @property
    def actions_check(self):
        return {
            CERT_DETAIL_ACTION.HOLD: self.check_hold,
            CERT_DETAIL_ACTION.UNHOLD: self.check_hold,
            CERT_DETAIL_ACTION.UPDATE: self.check_update,
            CERT_DETAIL_ACTION.REVOKE: self.check_revoke,  # crunch CERTOR-876 (Убрать после выкатки нового интерфейса)
        }

    def has_object_permission(self, request, view, obj):
        if request.method in permissions.SAFE_METHODS:
            return True

        if request.method == 'DELETE':
            return self.check_revoke_perm(request, view, obj)

        if request.method != 'POST':
            return True

        action = request.data.get('action')
        action_check = self.actions_check.get(action)
        if action_check is None:
            return True

        return action_check(request, view, obj)


class CanUseCA(permissions.BasePermission):
    def has_permission(self, request, view):
        if not request.data or 'ca_name' not in request.data:
            return True

        ca_name = request.data['ca_name']
        if ca_name not in settings.AVAILABLE_CA:
            return True

        ca_cls = get_ca_cls(ca_name)
        if ca_cls.PERMISSION_REQUIRED:
            return request.user.has_perm(ca_cls.PERMISSION_REQUIRED)

        return True


class CanRequestCertificate(permissions.BasePermission):
    def has_permission(self, request, view):
        if not request.data or 'type' not in request.data:
            return True

        cert_type = request.data['type']
        requester = request.user
        csr = request.data.get('request', None)
        common_name = request.data.get('common_name', None)
        ca_name = request.data.get('ca_name', None)
        # Маркет выписывает сертификаты 'zombie' для своих зомбиков CERTOR-1881
        if (
            cert_type == CERT_TYPE.ZOMBIE
            and
            (
                ZombieCertSerializer.is_request_for_market_zombie(common_name, csr)
                and
                requester.has_perm('core.can_issue_market_zombie_certificates')
            )
        ):
            return True

        if cert_type in CERT_TYPE.DEVICE_TYPES:
            # HelpDesk выписывает сертификаты для пользователей
            if requester.has_perm('core.can_issue_device_certificates'):
                return True
            # Внешние сотрудники HelpDesk выписывают сертификаты для внешних
            if requester.has_perm('core.can_issue_device_certificates_for_external'):
                from intranet.crt.api.v2.certificates.serializers.dispatch import get_serializer
                serializer = get_serializer(cert_type, ca_name)

                if cert_type == CERT_TYPE.ASSESSOR:
                    common_name = serializer.validate_common_name(common_name)

                actual_canonical_name = (serializer.get_common_name(csr) if csr else common_name)
                user = serializer.get_user_from_common_name(actual_canonical_name)
                return user.is_external_employee()

            return False

        elif cert_type == CERT_TYPE.BANK_PC:
            return requester.has_perm('core.can_issue_bank_pc_certificates')
        elif cert_type == CERT_TYPE.RC_SERVER:
            return requester.has_perm('core.can_issue_rc_server_certificates')
        elif cert_type == CERT_TYPE.YC_SERVER:
            return requester.has_perm('core.can_issue_yc_server_certificates')
        elif cert_type == CERT_TYPE.MDB:
            return requester.has_perm('core.can_issue_mdb_certificates')
        elif cert_type == CERT_TYPE.ZOMB_PC:
            return requester.has_perm('core.can_issue_zomb_pc_certificates')
        elif cert_type == CERT_TYPE.CLIENT_SERVER:
            return requester.has_perm('core.can_issue_client_server_certificates')
        elif cert_type == CERT_TYPE.BANK_CLIENT_SERVER:
            return requester.has_perm('core.can_issue_bank_client_server_certificates')
        elif cert_type == CERT_TYPE.HYPERCUBE:
            return (
                requester.has_perm('core.can_issue_hypercube_certificates')
                or (
                    request.data.get('desired_ttl_days') == 3
                    and requester.has_perm('core.can_issue_hypercube_certificates_3d')
                )
            )
        elif cert_type == CERT_TYPE.SDC:
            return request.user.has_perm('core.can_issue_sdc_certificates')
        elif cert_type == CERT_TYPE.TPM_SMARTCARD_1C:
            return request.user.has_perm('core.can_issue_tpm_smartcard_1c_certificates')
        elif cert_type == CERT_TYPE.POSTAMATE:
            return request.user.has_perm('core.can_issue_postamate_certificates')
        elif cert_type == CERT_TYPE.IMDM:
            return request.user.has_perm('core.can_issue_imdm_certificates')
        elif cert_type == CERT_TYPE.TEMP_PC:
            return request.user.has_perm('core.can_issue_temp_pc_certificates')
        elif cert_type == CERT_TYPE.VPN_1D:
            return request.user.has_perm('core.can_issue_vpn_1d_certificates')
        elif cert_type in CERT_TYPE.PUBLIC_REQUEST_TYPES:
            return True
        elif cert_type in CERT_TYPE.INACTIVE_TYPES:
            raise BadRequest('Cannot request certificate of inactive type')
        raise BadRequest('Unknown certificate type')


class CanRequestByCsrWithTagOids(permissions.BasePermission):
    message = 'Issuing a certificate with tag OIDs requires special permission'

    def csr_extensions_permitted(self, csr, cert_type):
        custom_extensions = get_x509_custom_extensions(PemCertificateRequest(csr).x509_object)

        if not custom_extensions:
            return True

        permitted_extensions = CERT_EXTENSION.PERMITTED_IN_PUBLIC_CSR.get(cert_type, {})
        forbidden_extensions = set(custom_extensions.keys()) - set(permitted_extensions.keys())

        if forbidden_extensions:
            self.message = 'CSR contains restricted extensions'
            return False

        for extension in custom_extensions.keys():
            value = force_text(custom_extensions[extension].value)
            values_set = set(value.split(','))
            if values_set - permitted_extensions[extension]:
                self.message = 'CSR extensions contains restricted values'
                return False

        return True

    def has_permission(self, request, view):
        if not request.data or 'request' not in request.data:
            return True

        csr = request.data['request']
        if csr and request_contains_custom_crt_extensions(csr):
            if request.user.has_perm('core.can_issue_certificate_with_tag_oids'):
                return True
            elif request.data.get('type') in CERT_EXTENSION.PERMITTED_IN_PUBLIC_CSR:
                return self.csr_extensions_permitted(csr, cert_type=request.data.get('type'))
            return False

        return True


class IsOwner(permissions.BasePermission):
    message = ugettext_lazy('You should be the owner of the certificate')

    def has_object_permission(self, request, view, obj):
        if (
            obj.abc_service_id and
            request.user.staff_groups.filter(
                abc_service=obj.abc_service_id,
                role_scope__in={ABC_CERTIFICATE_MANAGER_SCOPE, ABC_ADMINISTRATOR_SCOPE}
            ).exists()
        ):
            return True

        if obj.requester.username == request.user.username.lower():
            return True

        if obj.user and obj.user.username == request.user.username.lower():
            return True

        if (
            obj.type.name == CERT_TYPE.HOST
            and
            request.user.has_perm('core.is_responsible_for_any_host')
        ):
            return True, []

        return False
