from django.core.paginator import Page
from rest_framework import renderers

from intranet.crt.utils.iphone import get_iphone_exchange_profile
from intranet.crt.utils.ssl import create_pfx
from rest_framework.request import override_method, is_form_media_type
from rest_framework.exceptions import ParseError

ERROR_MESSAGE = 'Access denied'


class PEMRenderer(renderers.BaseRenderer):
    media_type = 'application/x-pem-file'
    format = 'pem'

    def render(self, data, accepted_media_type=None, renderer_context=None):
        if 'object' not in data:
            return ERROR_MESSAGE

        cert = data['object']
        parts = [cert.priv_key, cert.certificate]

        ca_chain_filename = cert.controller.ca_cls.get_chain_path(is_ecc=cert.is_ecc)
        with open(ca_chain_filename) as chain_file:
            parts.append(chain_file.read())

        parts = [_f for _f in parts if _f]
        return '\n'.join(parts)


class PFXRenderer(renderers.BaseRenderer):
    media_type = 'application/x-pkcs12'
    format = 'pfx'

    def render(self, data, accepted_media_type=None, renderer_context=None):
        if 'object' not in data:
            return ERROR_MESSAGE

        request = renderer_context['request']
        include_cacerts = int(request.GET.get('with_chain', '1'))

        db_certificate = data['object']
        password = request.GET.get('password', '7777777')

        cacerts_filename = db_certificate.controller.ca_cls.get_chain_path(is_ecc=db_certificate.is_ecc)
        return create_pfx(db_certificate, password, include_cacerts, cacerts_filename)


class IphoneRenderer(PFXRenderer):
    media_type = 'text/plain'
    format = 'iphoneexchange'

    def render(self, data, accepted_media_type=None, renderer_context=None):
        if 'object' not in data:
            return ERROR_MESSAGE

        pfx = super(IphoneRenderer, self).render(
            data=data,
            accepted_media_type=accepted_media_type,
            renderer_context=renderer_context,
        )

        request = renderer_context['request']
        pfx_password = request.GET.get('password', '7777777')

        return get_iphone_exchange_profile(data['object'], pfx, pfx_password)


class AlmostBrowsableAPIRenderer(renderers.BrowsableAPIRenderer):
    def render_form_for_serializer(self, serializer):
        if hasattr(serializer, 'initial_data'):
            serializer.is_valid()

        # ---- наши правки ----
        fields = list(serializer.fields.keys())
        for fieldname in fields:
            if fieldname in serializer.Meta.read_only_fields:
                serializer.fields.pop(fieldname)
        hosts = serializer.data.get('hosts')
        if isinstance(hosts, list):
            serializer._data['hosts'] = ','.join(hosts)
        # ---------------------

        form_renderer = self.form_renderer_class()
        return form_renderer.render(
            serializer.data,
            self.accepted_media_type,
            {'style': {'template_pack': 'rest_framework/horizontal'}}
        )

    def get_rendered_html_form(self, data, view, method, request):
        """
        Return a string representing a rendered HTML form, possibly bound to
        either the input or output data.

        In the absence of the View having an associated form then return None.
        """
        # See issue #2089 for refactoring this.

        # ------ немного изменений ------
        actual_method = request.method
        # -------------------------------

        serializer = getattr(data, 'serializer', None)
        if serializer and not getattr(serializer, 'many', False):
            instance = getattr(serializer, 'instance', None)
            if isinstance(instance, Page):
                instance = None
        else:
            instance = None

        # If this is valid serializer data, and the form is for the same
        # HTTP method as was used in the request then use the existing
        # serializer instance, rather than dynamically creating a new one.
        if request.method == method and serializer is not None:
            try:
                kwargs = {'data': request.data}
                # ----- начало блока с изменениями -----
                if request.method == 'GET' and 'type' in request.GET and 'ca_name' in request.GET:
                    kwargs['data'] = request.GET
                # ------ конец блока с измененями ------
            except ParseError:
                kwargs = {}
            existing_serializer = serializer
        else:
            kwargs = {}
            existing_serializer = None

        with override_method(view, request, method) as request:
            # ------ ещё немного изменений ------
            if method == 'POST' and actual_method == 'GET':
                request.data.update({
                    'type': request.query_params.get('type'),
                    'ca_name': request.query_params.get('ca_name'),
                })
            # -----------------------------------
            if not self.show_form_for_method(view, method, request, instance):
                return

            if method in ('DELETE', 'OPTIONS'):
                return True  # Don't actually need to return a form

            has_serializer = getattr(view, 'get_serializer', None)
            has_serializer_class = getattr(view, 'serializer_class', None)

            if (
                (not has_serializer and not has_serializer_class) or
                not any(is_form_media_type(parser.media_type) for parser in view.parser_classes)
            ):
                return

            if existing_serializer is not None:
                try:
                    return self.render_form_for_serializer(existing_serializer)
                except TypeError:
                    pass

            if has_serializer:
                if method in ('PUT', 'PATCH'):
                    serializer = view.get_serializer(instance=instance, **kwargs)
                else:
                    serializer = view.get_serializer(**kwargs)
            else:
                # at this point we must have a serializer_class
                if method in ('PUT', 'PATCH'):
                    serializer = self._get_serializer(view.serializer_class, view,
                                                      request, instance=instance, **kwargs)
                else:
                    serializer = self._get_serializer(view.serializer_class, view,
                                                      request, **kwargs)

            return self.render_form_for_serializer(serializer)
