import re

import constance

from django.conf import settings
from django.core.validators import EmailValidator
from django.utils.encoding import force_text
from rest_framework import serializers, relations
from rest_framework.relations import SlugRelatedField
from library.python.vault_client.errors import ClientError

from intranet.crt.api.v1.certificates.fields import DownloadField
from intranet.crt.constants import TAG_SOURCE
from intranet.crt.core.ca import registry
from intranet.crt.core.models import CertificateType, Certificate
from intranet.crt.core.utils import get_yav_client
from intranet.crt.exceptions import CsrMultipleOid
from intranet.crt.tags.models import CertificateTagRelation
from intranet.crt.users.models import CrtUser
from intranet.crt.utils.ssl import PemCertificateRequest, PemCertificate, get_x509_custom_extensions

COMMON_READONLY_FIELDS = (
    'common_name', 'request_id', 'added', 'end_date', 'issued', 'updated', 'revoked', 'revoked_by', 'held_by',
    'serial_number', 'priv_key_deleted_at', 'device_platform', 'error_message', 'status', 'unheld_by',
    'requester', 'username', 'url', 'download', 'download2', 'used_template', 'id',
    'is_reissue', 'uploaded_to_yav', 'yav_secret_version', 'requested_by_csr', 'is_ecc',
    'tags', 'manual_tags', 'builtin_tags',
)
EXPLICIT_READONLY_FIELDS = (
    'type', 'ca_name', 'desired_ttl_days', 'yav_secret_id',
)


def exclude(fields, *fields_to_exclude):
    a_set = set(fields_to_exclude)
    return tuple(name for name in fields if name not in a_set)


class CertificateToTagRelationSerializer(serializers.ModelSerializer):
    name = serializers.ReadOnlyField(source='tag.name')
    is_active = serializers.ReadOnlyField(source='tag.is_active')
    source = serializers.ReadOnlyField()

    class Meta(object):
        model = CertificateTagRelation
        fields = ('name', 'is_active', 'source')


class BaseMeta(object):
    model = Certificate
    read_only_fields = COMMON_READONLY_FIELDS
    fields = COMMON_READONLY_FIELDS + EXPLICIT_READONLY_FIELDS


class BaseCertSerializer(serializers.HyperlinkedModelSerializer):
    ca_name = serializers.ChoiceField(
        choices=[(name, name) for name in settings.AVAILABLE_CA],
        required=True,
    )
    type = SlugRelatedField(
        queryset=CertificateType.objects.filter(is_active=True),
        slug_field='name',
    )
    username = serializers.SlugRelatedField(
        queryset=CrtUser.objects.all(),
        slug_field='username',
        required=False,
    )
    requester = serializers.SlugRelatedField(
        queryset=CrtUser.objects.all(),
        slug_field='username',
        required=False,
    )
    held_by = serializers.SlugRelatedField(
        slug_field='username',
        read_only=True,
    )
    revoked_by = serializers.SlugRelatedField(
        slug_field='username',
        read_only=True,
    )
    unheld_by = serializers.SlugRelatedField(
        slug_field='username',
        read_only=True,
    )
    url = relations.HyperlinkedIdentityField(
        view_name='api:certificate-detail',
        lookup_field='pk',
        read_only=True,
    )
    download = DownloadField(view_name='api:certificate-download', format='pfx', force_format=True)
    download2 = DownloadField(view_name='api:certificate-download2', format='pfx')
    tags = CertificateToTagRelationSerializer(source='tag_relation', many=True, read_only=True)
    manual_tags = serializers.SerializerMethodField()
    builtin_tags = serializers.SerializerMethodField()

    common_name_pattern = re.compile(r'^([\w._-]+)@ld\.yandex\.ru$')
    organizational_unit_pattern = None
    email_bad_symbols = '/='
    email_validator = EmailValidator(message='Invalid email address')

    class Meta(BaseMeta):
        pass

    def __init__(self, instance=None, data=serializers.empty, **kwargs):
        if isinstance(data, dict):
            data = {k: v for k, v in data.items() if v != ''}

        super(BaseCertSerializer, self).__init__(instance=instance, data=data, **kwargs)

    @classmethod
    def check_email(cls, email):
        cls.email_validator(email)
        if any(char in email for char in cls.email_bad_symbols):
            raise serializers.ValidationError('Invalid email address')

    def get_manual_tags(self, obj):
        return sorted([rel.tag.name for rel in obj.tag_relation.all() if rel.source == TAG_SOURCE.MANUAL])

    def get_builtin_tags(self, obj):
        builtin_tags = ''
        if obj.certificate:
            try:
                custom_extensions = get_x509_custom_extensions(PemCertificate(obj.certificate).x509_object)
                custom_extensions_strings = []
                for key in sorted(custom_extensions.keys()):
                    custom_extensions_strings.append(
                        f'{key}: {force_text(custom_extensions[key].value)}'
                    )

                builtin_tags = '; '.join(custom_extensions_strings)
            except ValueError:
                pass
        return builtin_tags

    @classmethod
    def check_common_name(cls, common_name):
        match = cls.common_name_pattern.match(common_name)
        if match is None:
            raise serializers.ValidationError('Invalid common name')

    @classmethod
    def check_organizational_unit(cls, organizational_unit):
        match = cls.organizational_unit_pattern.match(organizational_unit)
        if match is None:
            raise serializers.ValidationError('Invalid organizational unit name')

    @classmethod
    def get_common_name(cls, csr):
        try:
            common_name = PemCertificateRequest(csr).common_name
        except ValueError:
            raise serializers.ValidationError('Cannot parse CSR')
        except CsrMultipleOid:
            raise serializers.ValidationError('Multiply Common Name oid')
        if common_name is None:
            raise serializers.ValidationError('В CSR не задан Common Name.')

        cls.check_common_name(common_name)

        return common_name

    @classmethod
    def get_organizational_unit(cls, csr, require=True):
        try:
            organizational_unit = PemCertificateRequest(csr).organizational_unit
        except CsrMultipleOid:
            raise serializers.ValidationError('Multiply Organizational Unit Name oid')
        if organizational_unit is None:
            if not require:
                return
            raise serializers.ValidationError('В CSR не задан Organizational Unit Name.')

        cls.check_organizational_unit(organizational_unit)

        return organizational_unit

    @classmethod
    def get_user_from_common_name(cls, common_name):
        match = cls.common_name_pattern.match(common_name)

        if match is None:
            raise serializers.ValidationError('Invalid common name')

        username = match.group(1)

        try:
            user = CrtUser.objects.get(username=username)
        except CrtUser.DoesNotExist:
            raise serializers.ValidationError('User "{}" does not exist'.format(username))

        if not user.is_active and not user.in_hiring:
            raise serializers.ValidationError('User "{}" is inactive'.format(username))

        return user

    @classmethod
    def get_email(cls, csr):
        try:
            email = PemCertificateRequest(csr).email_address
        except ValueError:
            raise serializers.ValidationError('Cannot parse CSR')
        except CsrMultipleOid:
            raise serializers.ValidationError('Multiply email oid')
        if email is None:
            raise serializers.ValidationError('В CSR не задан email.')

        cls.check_email(email)
        return email

    @classmethod
    def get_dn_str(cls, csr):
        subject = PemCertificateRequest(csr).x509_object.subject
        return ','.join(rdn.rfc4514_string() for rdn in subject.rdns[::-1])

    def validate_hosts(self, hosts):
        if hosts is None:
            hosts = []
        return hosts

    def check_readonly_fields(self, attrs):
        fields_to_display = set(self.Meta.fields)
        readonly_fields = (
                set(self.Meta.read_only_fields) |
                set(field for field in fields_to_display if self.fields[field].read_only)
        )
        writable_fields = fields_to_display - readonly_fields

        bad_fields = set(attrs.keys()) - writable_fields
        for field in bad_fields:
            if attrs[field]:
                message = 'Поле {0} запрещено для данного типа сертификатов.'.format(field)
                self._errors[field] = message
        return attrs

    def check_required_fields(self, attrs):
        required_fields = getattr(self.Meta, 'required', [])
        if not hasattr(self, '_errors'):
            self._errors = {}
        for name, field in self.fields.items():
            if not attrs.get(name) and (field.required or name in required_fields):
                self._errors.setdefault(name, [])
                self._errors[name].append('This field is required.')
        return attrs

    def check_ca_and_type_compatibility(self, attrs):
        ca_name = attrs['ca_name']
        supported_types = (
            registry.CA_REGISTRY
                .get(ca_name, {})
                .get('kwargs', {})
                .get('supported_types', set())
        )
        cert_type = attrs['type'].name
        if cert_type not in supported_types:
            raise serializers.ValidationError(f'Invalid {cert_type} cert for {ca_name}')
        return attrs
    
    def check_all_certs_validation(self, attrs):
        attrs = self.check_required_fields(attrs)
        attrs = self.check_readonly_fields(attrs)
        attrs = self.check_ca_and_type_compatibility(attrs)
        return attrs

    def check_yav_secret(self, secret_id):
        client = get_yav_client()
        try:
            client.get_secret(secret_id, return_raw=False)
        except ClientError:
            self._errors['yav_secret_id'] = \
                'Секрет с id {0} не существует или у пользователя {1} отсутствует к нему доступ'\
                .format(secret_id, settings.CRT_ROBOT)

    def validate(self, attrs):
        attrs = self.check_all_certs_validation(attrs)
        if 'request' not in attrs and 'yav_secret_id' in attrs:
            self.check_yav_secret(attrs['yav_secret_id'])
        if self.errors:
            raise serializers.ValidationError(self.errors)
        return attrs

    def save(self, *args, **kwargs):
        self.pre_save()

        super(BaseCertSerializer, self).save(*args, **kwargs)

        self.post_save()

        return self.instance

    def pre_save(self):
        user = self._context['request'].user
        self.validated_data['user'] = user
        self.validated_data['requester'] = user
        self.validated_data['requested_by_csr'] = 'request' in self.validated_data
        if user.username in constance.config.YAV_BLACKLIST:
            self.validated_data['uploaded_to_yav'] = None

    def post_save(self):
        pass

    def to_representation(self, instance):
        instance.username = instance.user

        return super(BaseCertSerializer, self).to_representation(instance)
