from django.conf import settings
from rest_framework import serializers, exceptions

from intranet.crt.api.v1.certificates.serializers.base import BaseCertSerializer
from intranet.crt.api.v1.certificates.serializers.specified import (
    BotikCertSerializer,
    ClientServerCertSerializer,
    BankClientServerCertSerializer,
    CourtecyVPNCertSerializer,
    HostsCertSerializer,
    ExternalHostsCertSerializer,
    ImdmCertSerializer,
    MobvpnCertSerializer,
    PcCertSerializer,
    BankPcCertSerializer,
    LinuxPcCertSerializer,
    LinuxTokenCertSerializer,
    AssessorCertSerializer,
    NinjaCertSerializer,
    NinjaExchangeCertSerializer,
    HypercubeCertSerializer,
    RcServerCertSerializer,
    Vpn1DCertSerializer,
    VpnTokenCertSerializer,
    YcServerCertSerializer,
    ZombieCertSerializer,
    PostamateCertSerializer,
    MdbCertSerializer,
    SdcCertSerializer,
    ZombPcCertSerializer,
    TpmSmartcard1CCertSerializer,
    TempPcCertSerializer,
)
from intranet.crt.constants import CERT_TYPE
from intranet.crt.core.ca import get_ca_cls


class CertificateSerializerDispatcher(serializers.Serializer):
    def to_representation(self, value):
        serializer = get_serializer(value.type.name, value.ca_name)
        return serializer(value, context=self.context).data


def get_serializer(cert_type_name, ca_name):
    if cert_type_name is None:
        return BaseCertSerializer

    if ca_name is None:
        raise exceptions.ParseError('ca_name field is required')

    if ca_name not in settings.AVAILABLE_CA:
        raise exceptions.ParseError(
            'Invalid ca_name {ca_name}. Available CAs: {available_cas}'
                .format(ca_name=ca_name, available_cas=', '.join(settings.AVAILABLE_CA))
        )

    if cert_type_name == CERT_TYPE.HOST and get_ca_cls(ca_name).IS_EXTERNAL:
        return ExternalHostsCertSerializer

    certificate_serializers = {
        CERT_TYPE.PC: PcCertSerializer,
        CERT_TYPE.BANK_PC: BankPcCertSerializer,
        CERT_TYPE.ZOMB_PC: ZombPcCertSerializer,
        CERT_TYPE.HOST: HostsCertSerializer,
        CERT_TYPE.COURTECY_VPN: CourtecyVPNCertSerializer,
        CERT_TYPE.LINUX_PC: LinuxPcCertSerializer,
        CERT_TYPE.LINUX_TOKEN: LinuxTokenCertSerializer,
        CERT_TYPE.MOBVPN: MobvpnCertSerializer,
        CERT_TYPE.BOTIK: BotikCertSerializer,
        CERT_TYPE.ASSESSOR: AssessorCertSerializer,
        CERT_TYPE.RC_SERVER: RcServerCertSerializer,
        CERT_TYPE.YC_SERVER: YcServerCertSerializer,
        CERT_TYPE.CLIENT_SERVER: ClientServerCertSerializer,
        CERT_TYPE.BANK_CLIENT_SERVER: BankClientServerCertSerializer,
        CERT_TYPE.NINJA: NinjaCertSerializer,
        CERT_TYPE.NINJA_EXCHANGE: NinjaExchangeCertSerializer,
        CERT_TYPE.HYPERCUBE: HypercubeCertSerializer,
        CERT_TYPE.VPN_1D: Vpn1DCertSerializer,
        CERT_TYPE.VPN_TOKEN: VpnTokenCertSerializer,
        CERT_TYPE.IMDM: ImdmCertSerializer,
        CERT_TYPE.ZOMBIE: ZombieCertSerializer,
        CERT_TYPE.POSTAMATE: PostamateCertSerializer,
        CERT_TYPE.MDB: MdbCertSerializer,
        CERT_TYPE.SDC: SdcCertSerializer,
        CERT_TYPE.TPM_SMARTCARD_1C: TpmSmartcard1CCertSerializer,
        CERT_TYPE.TEMP_PC: TempPcCertSerializer,
    }

    return certificate_serializers.get(cert_type_name, BaseCertSerializer)
