import datetime

from django.conf import settings
from django.db import transaction
from django.http import HttpResponseNotFound
from django.utils import timezone
from django.utils.encoding import force_text
from rest_framework import generics, response, exceptions, status

from intranet.crt.api.base.exceptions import ReplicationLag, NotFound
from intranet.crt.api.v1.certificates.filters import CertificateFilter, available_cas_uppercase_normalcase
from rest_framework.renderers import JSONRenderer
from django.utils.datastructures import MultiValueDict

from intranet.crt.api.v1.certificates import permissions, renderers
from intranet.crt.api.v1.certificates.forms import CertificateUpdateForm
from intranet.crt.api.v1.certificates.serializers.dispatch import CertificateSerializerDispatcher, get_serializer
from intranet.crt.constants import CERT_STATUS, ACTION_TYPE, CERT_DETAIL_ACTION, TAG_SOURCE, CERT_TYPE
from intranet.crt.core.ca.exceptions import ValidationCaError, CaError, Ca429Error
from intranet.crt.core.models import Certificate
from intranet.crt.core.query_parser import parse_search_query
from intranet.crt.tags.models import CertificateTagRelation, CertificateTag


@transaction.atomic
def update_certificate(request, certificate):
    """
    Функция для обновления abc_service, тегов (и в будущем других полей) от лица пользователя
    """
    update_form = CertificateUpdateForm(request.data)
    if not update_form.is_valid():
        raise exceptions.ValidationError(update_form.errors)

    fields_to_change = set(request.data.keys())

    if 'abc_service' in fields_to_change:
        new_service = update_form.cleaned_data['abc_service']
        new_service_id = new_service.external_id if new_service else None
        old_service_id = certificate.abc_service.external_id if certificate.abc_service else None
        if new_service_id != old_service_id:
            certificate.abc_service = new_service
            certificate.actions.create(type=ACTION_TYPE.CERT_CHANGE_ABC_SERVICE, user=request.user)
            certificate.save(update_fields=['abc_service'])

    if 'manual_tags' in fields_to_change:
        if certificate.type.name not in CERT_TYPE.TAGGABLE_TYPES:
            raise exceptions.ValidationError(
                'Tags cannot be edited for certificates of type "{}"'.format(certificate.type.name)
            )
        current = certificate.tag_relation.filter(source=TAG_SOURCE.MANUAL).values('id', 'tag__name', 'tag__id')
        current_tags = {rel['tag__name'] for rel in current}
        name_to_rel = {rel['tag__name']: rel['id'] for rel in current}
        new_tags = set(update_form.cleaned_data['manual_tags'].values_list('name', flat=True))

        tags_to_add = new_tags - current_tags
        tags_to_delete = current_tags - new_tags
        rels_to_delete = {name_to_rel[tag] for tag in tags_to_delete}

        if not update_form.cleaned_data['append_tags_only']:
            for rel in CertificateTagRelation.objects.filter(id__in=rels_to_delete):
                rel.delete()

        name_to_tag = {
            tag.name: tag
            for tag
            in CertificateTag.objects.active()
        }
        for tag_name in tags_to_add:
            CertificateTagRelation.objects.create(
                tag=name_to_tag[tag_name],
                certificate=certificate,
                source=TAG_SOURCE.MANUAL,
            )


class CertificateList(generics.ListCreateAPIView):
    model = Certificate
    permission_classes = [
        permissions.CanUseCA,
        permissions.CanRequestCertificate,
        permissions.CanRequestByCsrWithTagOids,
    ]
    serializer_class = CertificateSerializerDispatcher
    renderer_classes = [
        JSONRenderer,
        renderers.AlmostBrowsableAPIRenderer,
    ]

    def get_list_serializer(self, data):
        return self.serializer_class(
            data,
            many=True,
            context=self.get_serializer_context(),
        )

    def _get_without_meta(self, queryset):
        page_size = self.params.get('page_size') or settings.REST_FRAMEWORK['PAGE_SIZE']
        result = queryset[:int(page_size)]
        serializer = self.get_list_serializer(result)
        return response.Response(serializer.data)

    def _get_with_meta(self, queryset):
        page = self.paginate_queryset(queryset)
        result = page if page is not None else queryset
        serializer = self.get_list_serializer(result)
        if page is not None:
            return self.get_paginated_response(serializer.data)
        return response.Response(serializer.data)

    # Некрасивая копипаста из ListModelMixin с фиксированием сериализатора :/
    def get(self, request, *args, **kwargs):
        self._parse_params()
        queryset = self.filter_queryset(self.get_queryset())
        if self.params.get('no_meta'):
            return self._get_without_meta(queryset)
        return self._get_with_meta(queryset)

    def get_serializer_class(self):
        cert_type_name = self.request.data.get('type')
        ca_name = self.request.data.get('ca_name')
        return get_serializer(cert_type_name, ca_name)

    def _parse_params(self):
        if 'query' in self.request.query_params:
            query = self.request.query_params['query']
            params = parse_search_query(query)
            self.params = MultiValueDict(params.items())
        else:
            self.params = MultiValueDict(self.request.query_params.lists())

    def get_queryset(self):
        # Так мы можем получить ca_name в правильном регистре, если ca_name в списке допустимых
        # И нам не придется использовать UPPER при запросе к базе
        # В противном случае фильтр все равно бы отсек его
        original_ca_name = self.params.get('ca_name')
        if original_ca_name is not None:
            self.params['ca_name'] = available_cas_uppercase_normalcase.get(original_ca_name.upper())

        if 'serial_number' in self.params:
            self.params['serial_number'] = self.params['serial_number'].upper()

        self.params.pop('user', None)  # Используется только в v2

        # дефолтные фильтры следуют дальше
        if 'username' in self.params:
            if '__any__' in self.params['username']:
                del self.params['username']
        else:
            self.params.appendlist('username', self.request.user.username)

        if 'status' in self.params:
            if '__any__' in self.params['status']:
                del self.params['status']
        else:
            default_statuses = (
                CERT_STATUS.REQUESTED,
                CERT_STATUS.VALIDATION,
                CERT_STATUS.ISSUED,
                CERT_STATUS.HOLD,
            )
            for default_value in default_statuses:
                self.params.appendlist('status', default_value)

        if 'order_by' in self.params:
            ordering = self.params['order_by']
        else:
            ordering = '-added'

        query = (
            Certificate.objects
            .order_by(ordering)
            .select_related(
                'user',
                'requester',
                'revoked_by',
                'held_by',
                'unheld_by',
                'type',
                'abc_service',
            )
            .prefetch_related(
                'hosts',
                'hosts_to_approve',
                'approve_request',
            )
            .prefetch_tags(only_active=False)
        )

        return CertificateFilter(self.params, query).qs

    def perform_create(self, serializer):
        certificate = serializer.save()

        if certificate.is_async:
            return

        try:
            certificate.controller.issue()
        except ValidationCaError as error:
            raise exceptions.ParseError(force_text(error))

        if certificate.status == CERT_STATUS.ERROR:
            raise exceptions.APIException(detail=certificate.error_message)


class CertificateDetail(generics.RetrieveAPIView):
    model = Certificate
    queryset = Certificate.objects.all()
    serializer_class = CertificateSerializerDispatcher
    permission_classes = [permissions.CertificateActionsPermission]

    def do_hold(self, request, cert):
        cert.controller.hold(request.user, description='by api')

    def do_unhold(self, request, cert):
        cert.controller.unhold(request.user, description='by api')

    def do_update(self, request, cert):
        update_certificate(request, cert)

    def do_revoke(self, request, cert):
        cert.controller.revoke(request.user, description='by api')

    @property
    def action_methods(self):
        return {
            CERT_DETAIL_ACTION.HOLD: self.do_hold,
            CERT_DETAIL_ACTION.UNHOLD: self.do_unhold,
            CERT_DETAIL_ACTION.UPDATE: self.do_update,
            CERT_DETAIL_ACTION.REVOKE: self.do_revoke,  # crunch CERTOR-876 (Убрать после выкатки нового интерфейса)
        }

    def post(self, request, *args, **kwargs):
        action = request.data.get('action')
        handler = self.action_methods.get(action)
        if handler is None:
            raise exceptions.ParseError('Invalid action "{}"'.format(action))

        cert = self.get_object()
        handler(request, cert)

        return self.retrieve(request, *args, **kwargs)

    def delete(self, request, *args, **kwargs):
        cert = self.get_object()
        try:
            cert.controller.revoke(user=request.user, description='by api')
        except Ca429Error as exc:
            resp = response.Response(exc.get_message(), status=status.HTTP_429_TOO_MANY_REQUESTS)
            resp['Retry-After'] = exc.retry_after
            return resp
        except CaError as exc:
            return response.Response('CA returned error', status=status.HTTP_500_INTERNAL_SERVER_ERROR)

        return self.retrieve(request, *args, **kwargs)


class CertificateDownload(generics.GenericAPIView):
    model = Certificate
    queryset = Certificate.objects.all()
    renderer_classes = (renderers.PEMRenderer, renderers.PFXRenderer, renderers.IphoneRenderer)
    permission_classes = [permissions.IsOwner]

    REQUIRE_PRIVATE_KEY = {'pfx', 'iphoneexchange'}

    def get(self, request, *args, **kwargs):
        cert = self.get_object()
        if cert.status not in CERT_STATUS.RENDERABLE_STATUSES:
            now = timezone.now()
            if now - cert.added <= datetime.timedelta(seconds=settings.CRT_EXPECTED_REPLICATION_LAG):
                raise ReplicationLag(wait=settings.CRT_EXPECTED_REPLICATION_LAG)
            raise NotFound('Certificate status is %s, could not render pfx' % cert.status)

        render_format = request.accepted_renderer.format

        if render_format in self.REQUIRE_PRIVATE_KEY and not cert.priv_key:
            message = (
                'Невозможно создать {} файл для сертификата {}, приватный ключ не найден'
                .format(render_format.upper(), cert.id)
            )
            return HttpResponseNotFound(
                message.encode('utf-8')
            )

        default_filename = '{}.{}'.format(cert.common_name, render_format)
        filename = request.GET.get('filename', default_filename)
        content_disposition = 'attachment; filename="{}"'.format(filename)

        result = response.Response({'object': cert})
        result['Content-Disposition'] = content_disposition
        return result

    def handle_exception(self, exc):
        response = super(CertificateDownload, self).handle_exception(exc)
        if response.status_code == status.HTTP_404_NOT_FOUND:
            detail = 'NOT FOUND'
            if hasattr(response, 'data') and 'detail' in response.data:
                detail = response.data['detail']
            new_response = HttpResponseNotFound(detail)
            if response.get('retry-after'):
                new_response['retry-after'] = response['retry-after']
            return new_response
        return response
